 /**
 *
 * \file        HPRFGW_RFTrans.c
 *
 * \brief       This file contains the RF transmit task routines.
 *
 * \detail
 *
 * \author      Hazrat Shah
 *
 * \date        08/22/2006
 * \warning
 * \note
 * \review
 */
#include "hprfgw_config.h"
#if defined (OS_NUCLEUS) || defined (OS_WINCE)
	#include <string.h>
#endif
#include "hprfgw_rf802_15_4.h"
#include "hprfgw_rfSlaveIntToHost.h"
#include "hprfgw_rfResultCode.h"
#include "hprfgw_rfHeartbeat.h"
#include "hprfgw_rfRecMsg.h"
#include "hprfgw_rfMsgSystem.h"
#include "hprfgw_rfDesc.h"
#include "hprfgw_rfComLog.h"
#include "hprfgw_rfRec.h"
#include "hprfgw_rfApp.h"
#include "hprfgw_rfTrans.h"
#include "hprfgw_rfMsgSystem.h"
#include "hprfgw_rfDiagnostics.h"
//#include "hprfgw_rfMacPhyInit.h"

#ifdef	RF_NETWORK_MASTER_DEVICE
#include "masterlist.h"
#include "hprfgw_cresnet.h"
#endif //#ifdef	RF_NETWORK_MASTER_DEVICE

////////////////////////////////////////////////////////////////////////////////
//	DEFINITIONS
////////////////////////////////////////////////////////////////////////////////
#ifndef RF_TASKS_BASE_PRIORITY
 #if defined (OS_NUCLEUS)
  #pragma info "!!!!Pass in the RF_TASKS_BASE_PRIORITY using the make file!!!!"
 #endif
#endif
#define RFTRANS_TASK_PRIORITY   					RF_TASK_PRIORITY(RF_TASKS_BASE_PRIORITY, 2)
#define	RFTRANS_TASK_STACK 							(1024*32)
#if defined (OS_NUCLEUS)
 #define	RFTRANS_PIPE_TIMEOUT					500
#elif defined (OS_WINCE)
 #define	RFTRANS_PIPE_TIMEOUT					5000	/* In mS */
#elif defined (OS_LINUX)
 /* TODO: Arbitrarily chose this time. */
 #define    RFTRANS_PIPE_TIMEOUT                    5000
#endif

//#define	RFTRANS_CONGESTION_AVOIDANCE				

//	events flags for RF Device 1..255 pipes
#define	RFTRANS_EVENT_RFDEVICE_PIPE0_31				0x00000001
#define	RFTRANS_EVENT_RFDEVICE_PIPE32_63			0x00000002
#define	RFTRANS_EVENT_RFDEVICE_PIPE64_95			0x00000004
#define	RFTRANS_EVENT_RFDEVICE_PIPE96_127  			0x00000008
#define	RFTRANS_EVENT_RFDEVICE_PIPE128_159 			0x00000010
#define	RFTRANS_EVENT_RFDEVICE_PIPE160_191			0x00000020
#define	RFTRANS_EVENT_RFDEVICE_PIPE192_223			0x00000040
#define	RFTRANS_EVENT_RFDEVICE_PIPE224_255			0x00000080
#define	RFTRANS_EVENT_RFDEVICE_PIPE_MASK			0x000000FF
/*
#define	RFTRANS_EVENT_								0x00000200
#define	RFTRANS_EVENT_								0x00000400
#define	RFTRANS_EVENT_								0x00000800
#define	RFTRANS_EVENT_								0x00001000
#define	RFTRANS_EVENT_								0x00002000
#define	RFTRANS_EVENT_								0x00004000
#define	RFTRANS_EVENT_								0x00008000
#define	RFTRANS_EVENT_								0x00010000
#define	RFTRANS_EVENT_								0x00020000
#define	RFTRANS_EVENT_								0x00040000
#define	RFTRANS_EVENT_								0x00080000
*/

//	event flag for RF stack buffer avaliable
#define	RF_STACK_BUFFER_AVAILABLE_EVENT				0x00000001

enum	{
	FRAME_POST_TO_RFSTACK_SUCCESS=0,

	//	List of error codes for RFTransmit_PostFrameToRFStack()
	FRAME_POST_TO_RFSTACK_EXEC_ERROR=-1,
	FRAME_POST_TO_RFSTACK_ALLOCATION_ERROR=-2,
	FRAME_POST_TO_RFSTACK_FRAME_TOO_LARGE_ERROR=-3
};

#define	RFTRANS_TRIGGER_RF_TRANSMIT_TASK_MAINEVENTS		8
#define	RFTRANS_TRIGGER_RF_TRANSMIT_TASK_SUBEVENTS		32
// For WINCE only - total events
#define RFTRANS_TRIGGER_RF_TRANSMIT_EVENT_COUNT			(8*32)
#define RFTRANS_EVENTS_ARRAY_COUNT						((RFTRANS_TRIGGER_RF_TRANSMIT_EVENT_COUNT + 31) / 32)

typedef struct	{
#if defined (OS_NUCLEUS)
    NU_EVENT_GROUP TriggerRFTransmitTaskMainEvents;
	NU_EVENT_GROUP TriggerRFTransmitTaskSubEvents[RFTRANS_TRIGGER_RF_TRANSMIT_TASK_MAINEVENTS];
    NU_EVENT_GROUP rfstackbufferavailableevents;

    NU_TASK task;
	UINT8   stack[RFTRANS_TASK_STACK];
#elif defined (OS_WINCE)
	HANDLE				hRfTransmitEvent;
	UINT32				RfTransmitEventGroup[RFTRANS_EVENTS_ARRAY_COUNT];
	CRITICAL_SECTION	RfTransEventCriticalSection;
	HANDLE				hRfStackbufferavailableevents;
	HANDLE				task;
#elif defined (OS_LINUX)
	DM_RESOURCE_HANDLE	TriggerRFTransmitTaskMainEvents;
	DM_RESOURCE_HANDLE  TriggerRFTransmitTaskSubEvents[RFTRANS_TRIGGER_RF_TRANSMIT_TASK_MAINEVENTS];
	DM_RESOURCE_HANDLE  RfTransEventCriticalSection;
	DM_RESOURCE_HANDLE	hRfStackBufferAvailableEvents;
	DM_RESOURCE_HANDLE	task;
#endif
}	RFTRANSTASK;
RFTRANSTASK RFTransTask;

//	The PIB attributes values are read/written from/to the stack
typedef	struct	{
	UINT8 value[20];	//	todo: re-structure for each attribute par
}	TYPE_PIB_ATTRIBUTES;
TYPE_PIB_ATTRIBUTES PIBASet[gMPibAclEntrySecuritySuite_c-gMPibRole_c+2];
TYPE_PIB_ATTRIBUTES PIBAGet[gMPibAclEntrySecuritySuite_c-gMPibRole_c+2];

UINT32 LowPriorityMessagePostingDelay;

#ifdef	RFTRANS_CONGESTION_AVOIDANCE
	UINT32	ca_transmitpktcnt, ca_transmitpktdly, ca_receivedpktcnt, ca_CCAfailures, ca_transmitpktretry;	
#endif
	
////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////
extern INT8 *ConvertUIDtoString (UINT8 *puid, INT8 *pstr, UINT8 AddrMode);
#ifdef	RF_NETWORK_MASTER_DEVICE
extern UINT16 CresnetTxBuildErrorSerialFrameAndPostToPipe (INT8 *pSerial, UINT8 len);
#endif

////////////////////////////////////////////////////////////////////////////////
//	LOCAL FUNCTIONS
///////////////////////////////////////////////////////////////////////////////

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function posts the data frame to the RF stack buffer
 *
 * \detail
 *
 * \date        08/29/2006
 * \param	    *pBuf, len, index
 *
 *
 * \return		result
 *
 * \retval
 */
UINT16	RFTransmit_PostMCPPacketToRFStack (UINT8 index, UINT8 *pBuf, UINT8 len)	{
	nwkToMcpsMessage_t *pPacket = NULL;
	UINT16	result = FRAME_POST_TO_RFSTACK_SUCCESS;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = (g_pRFDeviceDescriptiveTable+index);
	UINT8	uid[UID_MAX_SIZE];
	UINT16  panid;
	UINT16	i, j, l;
	INT8	buf[600],buf1[50], buf2[50];
	BOOL isTraceEnabled;

#ifdef  RF_DBG_LOG
	TYPE_EXTERNAL_RF_PACKET_HEADER *ppkthdr = (TYPE_EXTERNAL_RF_PACKET_HEADER*)(pBuf+sizeof(TYPE_RF_PAYLOAD_HEADER));
#endif

	RFIntHostApp.pF_IsRFTxDebugTraceEnabled(&isTraceEnabled);
	RFReceive_DisableTimerInterrupt();
	pPacket = MSG_Alloc(sizeof(nwkToMcpsMessage_t)+len);
	RFReceive_EnableTimerInterrupt();
	if (pPacket != NULL)	{
		pdst->waitingforACK       = TRUE;
		pdst->resultcodeavailable = FALSE;
		pdst->ResultCodeTimedOut  =	FALSE;
		pdst->ResultCodeTimer	  =	RFReceive_GetLocalDeviceResultCodeTimeout();

		// Create an MCPS-Data Request message containing the UART data.
  		pPacket->msgType = gMcpsDataReq_c;

 	    // Create the header
		RFReceive_GetLocalDeviceUID (uid);

		FLib_MemCpy(&pPacket->msgData.dataReq.dstAddr,  (INT8 *) &pdst->IDInfo.uid, UID_MAX_SIZE);
  		panid = ConvUINTToLittleIndian(pdst->IDInfo.panid);
		FLib_MemCpy(&pPacket->msgData.dataReq.dstPanId, &panid, 					PANID_MAX_SIZE);

		FLib_MemCpy(&pPacket->msgData.dataReq.srcAddr,  uid,						UID_MAX_SIZE);
		panid = ConvUINTToLittleIndian(RFReceive_GetLocalDevicePanID());
		FLib_MemCpy(&pPacket->msgData.dataReq.srcPanId, &panid,						PANID_MAX_SIZE);

		pPacket->msgData.dataReq.srcAddrMode = RFReceive_GetLocalDeviceAddrMode();
  		pPacket->msgData.dataReq.msduLength  = len;
		pPacket->msgData.dataReq.dstAddrMode = pdst->IDInfo.addrmode;

		if (pdst->TxBuf.DataHold.inthdr.hdroptions1 & RF_DEVICE_HEADER_OPTION1_BROADCAST)	{
			//	Broadcast to all devices (limitations are that all devices slaves, coordinator and possibly MNET devices receives this packet)
			memset (&pPacket->msgData.dataReq.dstAddr, 0xFF, UID_MAX_SIZE);
			pPacket->msgData.dataReq.dstAddrMode = gAddrModeShort_c;
			memset (&pPacket->msgData.dataReq.dstPanId, 0xFF, 2);

			//	Broadcast to PAN coordinators only (limitations are that the PANID of coordinators must be fixed and this broadcast is
			//	ignored by the slaves devices)
			//pPacket->msgData.dataReq.dstAddrMode = gAddrModeNone_c;
			//pPacket->msgData.dataReq.dstPanId    =

			//	No ack on broadcast
			pPacket->msgData.dataReq.txOptions = 0;
			//	On broadcasted packets do not wait for ACK
			//pdst->waitingforACK   = FALSE;
			//pdst->ResultCodeTimer =	0;

		}	else if ((pdst->TxBuf.DataHold.inthdr.hdroptions1 & RF_DEVICE_HEADER_OPTION1_REQ_ACK) && (((pdst->IDInfo.addrmode == gAddrModeLong_c) || (pdst->IDInfo.addrmode == gAddrModeShort_c))))	{
			pPacket->msgData.dataReq.txOptions = gTxOptsAck_c;
		}

  		//	Give the data packet a handle. The handle is returned in the MCPS-Data Confirm message.
  		pPacket->msgData.dataReq.msduHandle = index;

		//	Copy packet data
		FLib_MemCpy(&pPacket->msgData.dataReq.msdu, pBuf, len);

		if (isTraceEnabled)	{
			ConvertUIDtoString((UINT8*) &pPacket->msgData.dataReq.srcAddr, buf1, pPacket->msgData.dataReq.srcAddrMode);
			ConvertUIDtoString((UINT8*) &pPacket->msgData.dataReq.dstAddr, buf2, pPacket->msgData.dataReq.dstAddrMode);
			RFIntHostApp.pF_ConsolePrintf ("MCP TX: index[%x] handle[%02x] src[%s %02x%02x %x] dst[%s %02x%02x %x] len[%02x] opt[%x] plhdr[%02x%02x%02x%02x] ", index, pPacket->msgData.dataReq.msduHandle,
				buf1, pPacket->msgData.dataReq.srcPanId[0], pPacket->msgData.dataReq.srcPanId[1], pPacket->msgData.dataReq.srcAddrMode,
				buf2, pPacket->msgData.dataReq.dstPanId[0], pPacket->msgData.dataReq.dstPanId[1], pPacket->msgData.dataReq.dstAddrMode,
				pPacket->msgData.dataReq.msduLength, pPacket->msgData.dataReq.txOptions,
				pPacket->msgData.dataReq.msdu[0], pPacket->msgData.dataReq.msdu[1], pPacket->msgData.dataReq.msdu[2], pPacket->msgData.dataReq.msdu[3]);

			RFIntHostApp.pF_ConsolePrintf ("data[");
			for (i=4, buf[0]=0, l=pPacket->msgData.dataReq.msdu[i]+1, j=0; i<pPacket->msgData.dataReq.msduLength; i++)	{
				j += sprintf(&buf[j], "%02x", pPacket->msgData.dataReq.msdu[i]);
				if (--l == 0)	{
					l = pPacket->msgData.dataReq.msdu[i+1]+1;
					j += sprintf (&buf[j], " ");
				}
			}
			RFIntHostApp.pF_ConsolePrintf ("%s]\r\n", buf);
		}

		// Done before MSG_Send so that we get the variable before the free.
        pdst->COMLog.txpacketpayloadbytecount  += (pPacket->msgData.dataReq.msduLength);

        //	Send the Data Request to the MCPS
		RFReceive_DisableTimerInterrupt();
  		(void)NWK_MCPS_SapHandler(pPacket);
		RFReceive_EnableTimerInterrupt();

		// The buffer will be released by the stack

  		//	Frame posted successfully
  		result = FRAME_POST_TO_RFSTACK_SUCCESS;
		//	Total number of packets posted for transmission
		pdst->COMLog.txpacketcount	   += 1;
		//	packet len: preamble (4B) + frame control (2B)+ DesPanID (2B) + DestAddr (8B) + SrcPanID (2B) + SrcAddr (8B) + PayLoad (xB) + CheckSum (2B)
		// assumption the src and dst addr is always in longaddr mode
		pdst->COMLog.txpacketoverheadbytecount += 18;
		//	Log packet post time
		if (!pdst->txpacketretrycntr)	{	pdst->COMLog.txpacketposttime = OS_RetrieveClock();	}
		//	turn on the transmit packet indicator
		RFIntHostApp.pF_RFTransmitIndicatorFlash();

#ifdef	RF_DBG_LOG
		RFComLog_LogEvent (RF_DBG_MCP_TX_PACKET, index, 0, 0, ppkthdr->type, pdst->COMLog.txpacketcount, OS_RetrieveClock());
#endif

	}	else	{
		//	buffer could not be allocated
		result = FRAME_POST_TO_RFSTACK_ALLOCATION_ERROR;
	}

	return (result);
}


/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function posts an ASP frame to the RF stack buffer
 *
 * \detail
 *
 * \date        08/29/2006
 *
 * \param	    UINT8 *pBuf, UINT8 len
 *
 *
 * \return		UINT16 result
 *
 * \retval
 */
UINT16	RFTransmit_PostASPPacketToRFStack (UINT8 index, UINT8 *pBuf, UINT8 len)	{
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = (g_pRFDeviceDescriptiveTable+index);
	UINT16 res = FRAME_POST_TO_RFSTACK_SUCCESS;
	appToAspMsg_t Packet;
	UINT32	i;
	INT8	buf[300];
	UINT16	l;
	UINT8	status = 0;
	BOOL isTraceEnabled;

	RFIntHostApp.pF_IsRFTxDebugTraceEnabled(&isTraceEnabled);

	if	(len <= sizeof(appToAspMsg_t))	{
		//	Copy message to allocated buffer
		FLib_MemCpy(&Packet, pBuf, len);

		//	Send the ASP  Request to the ASP layer
		RFReceive_DisableTimerInterrupt();
		status = APP_ASP_SapHandler(pBuf);
		RFReceive_EnableTimerInterrupt();

		if(status == gSuccess_c)	{
	      	res = FRAME_POST_TO_RFSTACK_SUCCESS;
			//	Wait for confirmation before posting the next frame
			pdst->waitingforACK       = FALSE;
			pdst->resultcodeavailable = FALSE;
			pdst->ResultCodeTimedOut  =	FALSE;
			pdst->ResultCodeTimer	  =	0;

			pdst->COMLog.txpacketcount++;

			if (isTraceEnabled)	{
				l = sprintf(buf, "\r\n TXASP [");
				for (i=0; i<len; i++)	{	sprintf ((INT8*)&buf[l+i*2], "%02x",  *(pBuf+i));	}
				RFIntHostApp.pF_ConsolePrintf ("%s]\r\n", buf);
			}
#ifdef	RF_DBG_LOG
		RFComLog_LogEvent (RF_DBG_ASP_TX_PACKET, index, g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.type, g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype, 0, pdst->COMLog.txpacketcount, OS_RetrieveClock());
#endif
		}	else	{
			if (isTraceEnabled)	{
				RFIntHostApp.pF_ConsolePrintf ("ASP msg posting error: status[%02x], index[%02x]\r\n", status, index);
			}
		#ifdef	RF_MSG_ERR_LOG
				RFComLog_PutRFDeviceMsgError (index, g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.type,
					g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype, g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.req,
					RF_COMERROR_LOG_INIT_POSTFUNC, status);
		#endif	
			res = FRAME_POST_TO_RFSTACK_EXEC_ERROR;
		}
	}	else	{
		if (isTraceEnabled)	{
			RFIntHostApp.pF_ConsolePrintf ("ASP msg posting error frame too big: status[%02x] index[%02x] len[%02x]\r\n", status, index, len);
		}
		//	Frame too large
		res = FRAME_POST_TO_RFSTACK_FRAME_TOO_LARGE_ERROR;
	}

	//	make ASP packets call-back (ASP packets are syncronous)

	return (res);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function posts an MLM frame to the RF stack buffer
 *
 * \detail
 *
 * \date        09/13/2006
 *
 * \param	    UINT8 index, UINT8 *pBuf, UINT8 len
 *
 *
 * \return		UINT16 result
 *
 * \retval
 */
UINT16	RFTransmit_PostMLMEPacketToRFStack (UINT8 index, UINT8 *pBuf, UINT8 len)	{
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = (g_pRFDeviceDescriptiveTable+index);
	UINT16	res = FRAME_POST_TO_RFSTACK_SUCCESS;
	mlmeMessage_t *pPacket, *pP;
	UINT32	i;
	INT8	buf[300];
	UINT16	l;
	UINT8	status = 0;
	BOOL	isTraceEnabled;

	RFIntHostApp.pF_IsRFTxDebugTraceEnabled(&isTraceEnabled);

	if (len <= sizeof(mlmeMessage_t))	{

		//	Allocate buffer for mlmMessage
		RFReceive_DisableTimerInterrupt();
		pPacket = MSG_AllocType(mlmeMessage_t);
		RFReceive_EnableTimerInterrupt();

		if (pPacket != NULL)	{
			//	Copy message to allocated buffer
			FLib_MemCpy(pPacket, pBuf, len);

			RFReceive_DisableTimerInterrupt();
			status = NWK_MLME_SapHandler(pPacket);
			RFReceive_EnableTimerInterrupt();

			//	Send the MLME Request to the MLME layer
			if(status == gSuccess_c)	{
	      		res = FRAME_POST_TO_RFSTACK_SUCCESS;
				//	Wait for confirmation before posting the next frame
				pdst->waitingforACK       = TRUE;
				pdst->resultcodeavailable = FALSE;
				pdst->ResultCodeTimedOut  =	FALSE;
				pdst->ResultCodeTimer	  =	RFReceive_GetLocalDeviceResultCodeTimeout();

				pdst->COMLog.txpacketcount++;

				if (isTraceEnabled)	{
					l = sprintf(buf, "\r\n TXMLME [");
					for (i=0; i<len; i++)	{
						sprintf ((INT8*)&buf[l+i*2], "%02x",  *(pBuf+i));
					}
					RFIntHostApp.pF_ConsolePrintf ("%s]\r\n", buf);
				}

#ifdef	RF_DBG_LOG
				RFComLog_LogEvent (RF_DBG_MLME_TX_PACKET, index, g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.type, g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype, 0, pdst->COMLog.txpacketcount, OS_RetrieveClock());
#endif

			}	else	{
				//	Packet was not executed, Log error code
				if (isTraceEnabled)	{
					RFIntHostApp.pF_ConsolePrintf ("MLME msg posting error: status[%02x], index[%02x]\r\n", status, index);
				}
				#ifdef	RF_MSG_ERR_LOG
					RFComLog_PutRFDeviceMsgError (index, g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.type,
						g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype, g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.req,
						RF_COMERROR_LOG_INIT_POSTFUNC, status);
				#endif
				res = FRAME_POST_TO_RFSTACK_EXEC_ERROR;
			}

			pP = (mlmeMessage_t*) pBuf;

			//	De-allocate memory for certain MLME messsages
			if ((pP->msgType == gMlmeSetReq_c) || (pP->msgType == gMlmeResetReq_c) || (pP->msgType == gMlmeGetReq_c) || ((INT16)res == FRAME_POST_TO_RFSTACK_EXEC_ERROR))	{

				//	Release memory
				RFReceive_DisableTimerInterrupt();
				MSG_Free(pPacket);
				RFReceive_EnableTimerInterrupt();

				//	Confirmation is not sent for these commands
				pdst->waitingforACK   = FALSE;
				pdst->ResultCodeTimer = 0;

				//	make MLME packet call-back (these MLME packet types are executed syncronously)

			}
		}	else	{
			if (isTraceEnabled)	{
				RFIntHostApp.pF_ConsolePrintf ("MLME msg posting buffer allocation error: index[%02x] len[%02x]\r\n", index, len);
			}
			//	buffer allocation error
			res = FRAME_POST_TO_RFSTACK_ALLOCATION_ERROR;
		}
	}	else	{
		if (isTraceEnabled)	{
			RFIntHostApp.pF_ConsolePrintf ("MLME msg posting frame too big error: status[%02x] index[%02x] len[%02x]\r\n", status, index, len);
		}
		//	Frame is too large
		res = FRAME_POST_TO_RFSTACK_FRAME_TOO_LARGE_ERROR;
	}

	return (res);
}


/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function posts an MLM frame to the RF stack buffer
 *
 * \detail
 *
 * \date        09/13/2006
 *
 * \param	    UINT8 index, UINT8 *pBuf, UINT8 len
 *
 *
 * \return		UINT16 result
 *
 * \retval
 */
UINT16	RFTransmit_PreprocessAndPostMLMEPacket (UINT8 index, UINT8 *pBuf, UINT8 len)	{
	UINT16	res = FRAME_POST_TO_RFSTACK_SUCCESS;
	UINT8	att;
	mlmeMessage_t Msg;
	mlmeMessage_t *pm = (mlmeMessage_t*) pBuf;
	UINT8	i;

	//	Special handling for channel change command
	if ((pm->msgType == gMlmeSetReq_c) && (pm->msgData.setReq.pibAttribute == gMPibLogicalChannel_c))	{
		//	Note: The following frames are executed syncronously, the confirmation is returned when the command is executed

		//	Clear RxOnIdle
		att = FALSE;
		Msg.msgType = gMlmeSetReq_c;
		Msg.msgData.setReq.pibAttribute      = gMPibRxOnWhenIdle_c;
		Msg.msgData.setReq.pibAttributeValue = (UINT8*) &att;
		RFTransmit_PostMLMEPacketToRFStack (index, (UINT8*) &Msg, sizeof(mlmeMessage_t));

		//	Important note: MCP Descriptive entries waiting for result code are set to avaliable and the result is set to NAK
		//	to forces a retransmission of pending packets. However, queues received packets are dropped.
		//	It is recommended that an update request be issued after a channel change command
		for (i=0; i<g_MaxSupportedRFDevices; i++)	{
			if (g_pRFDeviceDescriptiveTable[i].waitingforACK)	{
				g_pRFDeviceDescriptiveTable[i].resultcodeavailable = TRUE;
				g_pRFDeviceDescriptiveTable[i].resultcode = gNoAck_c;
			}
		}
		//	Clear pending MLME request (not likely, but clear any way)
		if (g_pRFDeviceDescriptiveTable[RFDesc_GetMLMEDescriptiveTableIndex()].waitingforACK)	{
			g_pRFDeviceDescriptiveTable[RFDesc_GetMLMEDescriptiveTableIndex()].resultcodeavailable = TRUE;
			g_pRFDeviceDescriptiveTable[RFDesc_GetMLMEDescriptiveTableIndex()].resultcode = gNoAck_c;
		}
		//	Clear pending ASP request (not likely, but clear any way)
		if (g_pRFDeviceDescriptiveTable[RFDesc_GetASPDescriptiveTableIndex()].waitingforACK)	{
			g_pRFDeviceDescriptiveTable[RFDesc_GetASPDescriptiveTableIndex()].resultcodeavailable = TRUE;
			g_pRFDeviceDescriptiveTable[RFDesc_GetASPDescriptiveTableIndex()].resultcode = gNoAck_c;
		}
		//	flush pipes


		//	Clear stack MSG buffers
		RFReceiver_RFStackMsgQueInit();

		//	Wait

		//	Post channel command
		RFTransmit_PostMLMEPacketToRFStack (index, pBuf, len);

		//	Set RxOnIdle
		att = TRUE;
		Msg.msgType = gMlmeSetReq_c;
		Msg.msgData.setReq.pibAttribute      = gMPibRxOnWhenIdle_c;
		Msg.msgData.setReq.pibAttributeValue = (UINT8*) &att;
		RFTransmit_PostMLMEPacketToRFStack (index, (UINT8*) &Msg, sizeof(mlmeMessage_t));

		g_pRFDeviceDescriptiveTable[index].waitingforACK = FALSE;
		res = FRAME_POST_TO_RFSTACK_SUCCESS;
	}	else	{
		res = RFTransmit_PostMLMEPacketToRFStack (index, pBuf, len);
	}

	return (res);
}


/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function make sure that the network is not congested 
 *				In case of congestion it takes corrective action.	 
 *				
 *
 * \detail
 * \date        10/01/2007
 *
 * \param	    None
 *
 * \return		None
 *
 * \retval
 */
void	RFTransmit_avoidCongestion (UINT8 index)	{
#ifdef	RFTRANS_CONGESTION_AVOIDANCE
	UINT32 dly;

	ca_transmitpktcnt++;
	dly = (((Rfdiags_NumCCAFailures-ca_CCAfailures) + (Rfdiags_LowLevelRetryPacketCnt-ca_transmitpktretry))*2)/10;	  
	if ((ca_transmitpktcnt>20) && (((Rfdiags_NumCCAFailures-ca_CCAfailures)+(Rfdiags_LowLevelRetryPacketCnt-ca_transmitpktretry)) > 5))	{
		dly &= (1+(RFAPP_GetRandomNum()&3));		
//ConsolePrintf ("CA1 %02x\r\n", );	
	}	else if (dly && (((Rfdiags_NumCCAFailures-ca_CCAfailures)+(Rfdiags_LowLevelRetryPacketCnt-ca_transmitpktretry)) > 10))	{
		if (dly > 7)	{	dly = 7;	}			
//ConsolePrintf ("CA2 %02x\r\n", );
	}	else	{
		dly = 0;
	}

	if (dly)	{
		OS_Sleep(dly&15);
		ca_transmitpktcnt = 0;
		ca_transmitpktdly = OS_RetrieveClock();
		ca_CCAfailures = Rfdiags_NumCCAFailures;
		ca_transmitpktretry = Rfdiags_LowLevelRetryPacketCnt;	
		g_pRFDeviceDescriptiveTable[index].COMLog.txpacketcongavoidance++;
	}

	if ((OS_RetrieveClock()-ca_transmitpktdly) > 50)	{
		ca_transmitpktcnt	= 0;
		ca_transmitpktdly	= OS_RetrieveClock();
		ca_CCAfailures		= Rfdiags_NumCCAFailures;
		ca_transmitpktretry = Rfdiags_LowLevelRetryPacketCnt;	
	}		
#endif
}
	
/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function posts the frame to the RF stack buffer
 *
 * \detail
 *
 * \date        08/29/2006
 * \param	    index
 *
 *
 * \return		result
 *
 * \retval
 */
UINT16	RFTransmit_PostPacketToRFStack (UINT8 index)	{
	UINT32 events;
	UINT16 res = FRAME_POST_TO_RFSTACK_SUCCESS;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = (g_pRFDeviceDescriptiveTable+index);
	BOOL isTraceEnabled;

	HPRF_DEBUG(DmConsolePrintf("=========== %s =========== index = %u\n", __FUNCTION__, index);)
	RFIntHostApp.pF_IsRFTxDebugTraceEnabled(&isTraceEnabled);

	do	{
#if defined (OS_NUCLEUS)
		//	Clear RF stack buffer avaliable event flags
		NU_Retrieve_Events(&RFTransTask.rfstackbufferavailableevents, RF_STACK_BUFFER_AVAILABLE_EVENT, NU_OR_CONSUME, &events, NU_NO_SUSPEND);
#elif defined (OS_WINCE)
		//	Clear RF stack buffer avaliable event flags
		events = (UINT32) WaitForSingleObject(RFTransTask.hRfStackbufferavailableevents, OS_NO_SUSPEND);
#endif
		events = OsWaitEvent(RFTransTask.hRfStackBufferAvailableEvents, OS_NO_WAIT);
		switch (pdst->TxBuf.DataHold.inthdr.type)	{
			case RF_PACKET_TYPE_MLM_REQUEST:
				//	Note: For the MLME packets there is no payloader so, the data bytes are loaded starting at the payloader header section
				res = RFTransmit_PreprocessAndPostMLMEPacket (index, (UINT8 *) &pdst->TxBuf.Transmit.payld.hdr, pdst->TxBuf.Transmit.len);
				break;

			case RF_PACKET_TYPE_ASP_REQUEST:
				//	Note: For the ASP packets there is no payloader so, the data bytes are loaded starting at the payloader header section
				res = RFTransmit_PostASPPacketToRFStack (index, (UINT8 *) &pdst->TxBuf.Transmit.payld.hdr, pdst->TxBuf.Transmit.len);
				break;

			case RF_PACKET_TYPE_MCP_REQUEST:
				//	Note: For the MCP packets the data and header are separate, the len contains number of bytes in the data buffer
			    HPRF_DEBUG(DmConsolePrintf("===== %s ===== index = %u len = %u\n", __FUNCTION__, index, pdst->TxBuf.Transmit.len+sizeof(TYPE_RF_PAYLOAD_HEADER));)
				res = RFTransmit_PostMCPPacketToRFStack (index, (UINT8 *) &pdst->TxBuf.Transmit.payld.hdr, pdst->TxBuf.Transmit.len+sizeof(TYPE_RF_PAYLOAD_HEADER));
	#ifndef	RF_NETWORK_MASTER_DEVICE
				//	Reload the heart beat timer, regardless of whether the current packet was posted this could be because, the buffer is full
				//	It does not matter because, any packet from this device will refresh the heart beat timeout in the gateway
				//	this also, eliminates posting of heart beat packets to the buffer if it happens to be full
				RFHeartBeat_ReTriggerTimer (DEVICE_DESCRIPTIVE_TABLE_INDEX);
	#endif
				RFTransmit_avoidCongestion (index);
				break;
		}

		if 			(res == (UINT16) FRAME_POST_TO_RFSTACK_ALLOCATION_ERROR)	{
			if (isTraceEnabled)	{
				RFIntHostApp.pF_ConsolePrintf ("RF transmit stack buffer full\r\n");
			}
			//	Wait on	rf stack buffer avaliable event
#if defined (OS_NUCLEUS)
			if (NU_Retrieve_Events(&RFTransTask.rfstackbufferavailableevents, RF_STACK_BUFFER_AVAILABLE_EVENT, NU_OR_CONSUME, &events, NU_SUSPEND) == NU_SUCCESS)	{
#elif defined (OS_WINCE)
			if (WAIT_OBJECT_0 == WaitForSingleObject(RFTransTask.hRfStackbufferavailableevents, OS_SUSPEND))	{
#endif
			if(OsWaitEvent(RFTransTask.hRfStackBufferAvailableEvents, OS_WAIT_FOREVER)) {
				if (isTraceEnabled)	{
					//	buffer avaliabe event was received
					RFIntHostApp.pF_ConsolePrintf ("RF transmit stack buffer avaliable\r\n");
				}
			}
			else
			{
#if defined (OS_NUCLEUS)
				RFIntHostApp.pF_ErrorPrintf ("RFTransmit_PostPacketToRFStack: Fail waiting");
#elif defined (OS_WINCE)
				RFIntHostApp.pF_ErrorPrintf ("RFTransmit_PostPacketToRFStack: Fail waiting: %d", GetLastError());
#endif
				DmConsolePrintf("RFTransmit_PostPacketToRFStack: Fail waiting\n");
			}
		}
	}	while (res == (UINT16) FRAME_POST_TO_RFSTACK_ALLOCATION_ERROR);

	return (res);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function posts the frame to the data hold buffer
 *
 * \detail		This function copies the frame from temp buffer to data hold buffer.
 *				If there are existing frames in the data hold buffer and there is space for the new
 *				frame it will be concatenated given the frame transmit (ack/noack, broadcast) options are the same
 *				Larger frames are also, segmented by this function into smaller pieces and placed into the data hold
 *				buffer to be transmitted a segment at a time.
 *				The segmentation information is stored in the temp buffers so, the calling function should
 *				not store new data into temp buffer unitl it's fully emptied.
 *				(Note: This function handle storage for both low and high priority buffers, the only issue to
 *				be addressed is the syncronization of tx sequence counts)
 *
 *
 * \date        09/15/2006
 *
 * \param	    index
 *
 * \return		result
 *
 * \retval
 */
static UINT16 RFTransmit_BuildPacket (UINT8 index)	{
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = (g_pRFDeviceDescriptiveTable+index);
	TYPE_EXTERNAL_RF_PACKET_HEADER pktheader;
	UINT8	loop = TRUE;
	UINT8	buildpacket = FALSE;
	UINT16  len = 0;
	UINT16  transmitbufused;
	UINT32	cbcnt = 0;
	UINT32	hldlen = 0;
#if defined (OS_WINCE)
	DWORD   dwFlags;
#endif

	//	Prepare the MCP packet transmit buffer payload header (There is a single header for payload)
	pdst->TxBuf.Transmit.payld.hdr.linkquality = pdst->LQ.linkqualityAtLocalDevice;
	pdst->TxBuf.Transmit.payld.hdr.sequence  = pdst->txsequence;
	pdst->TxBuf.Transmit.payld.hdr.signature = HPRFGW_PACKET_SIGNATURE;	
	HPRF_HTONS(pdst->TxBuf.Transmit.payld.hdr.signature);
	//	Clear transmit packet length
	pdst->TxBuf.Transmit.len = 0;

	do	{
		if (pdst->TxBuf.DataHold.len)	{
			switch (pdst->TxBuf.DataHold.inthdr.type)	{
				case	RF_PACKET_TYPE_MCP_REQUEST:
					//	MCP data packets concatenation and segmentation
					if (pdst->TxBuf.DataHold.len > (RF_DEVICE_TRANSMIT_BUFFER_SIZE-sizeof(TYPE_EXTERNAL_RF_PACKET_HEADER)))	{
						//	Make sure there is enough room in the transmit buffer
						if ((RF_DEVICE_TRANSMIT_BUFFER_SIZE-pdst->TxBuf.Transmit.len) > sizeof(TYPE_EXTERNAL_RF_PACKET_HEADER))	{
							//	compute packet segment information if first segment
							if (pdst->TxBuf.DataHold.segmentno <= 1)	{
								//	calculate segments for large packets
								transmitbufused = pdst->TxBuf.Transmit.len;
								for (len=pdst->TxBuf.DataHold.len, pdst->TxBuf.DataHold.totalsegment = 1; len > (RF_DEVICE_TRANSMIT_BUFFER_SIZE-(sizeof(TYPE_EXTERNAL_RF_PACKET_HEADER)+transmitbufused)); pdst->TxBuf.DataHold.totalsegment +=1)	{
									len -= (RF_DEVICE_TRANSMIT_BUFFER_SIZE-(sizeof(TYPE_EXTERNAL_RF_PACKET_HEADER)+transmitbufused));
									transmitbufused = 0;
								}
								pdst->TxBuf.DataHold.index = 0;
								//	keep track of segmented packets
								pdst->COMLog.txpacketsegmented++;
							}
							//	Set segment length
							len = (RF_DEVICE_TRANSMIT_BUFFER_SIZE-(sizeof(TYPE_EXTERNAL_RF_PACKET_HEADER)+pdst->TxBuf.Transmit.len));
							//	raise flag to build packet header and data
							buildpacket = TRUE;
						}
						loop = FALSE;
					}	else	{
						//	Check if packet fits in the transmit buffer
						if ((pdst->TxBuf.DataHold.len+sizeof(TYPE_EXTERNAL_RF_PACKET_HEADER)) <= (RF_DEVICE_TRANSMIT_BUFFER_SIZE-pdst->TxBuf.Transmit.len))	{
							len  = pdst->TxBuf.DataHold.len;
							buildpacket = TRUE;
						}	else {
							loop = FALSE;
						}
					}

					if (buildpacket)	{
						buildpacket = FALSE;

						//	If a packet was successfully stored in the transmit buffer then increment the tx sequence count
						if (pdst->TxBuf.Transmit.len == 0)	{								
							if (pdst->TxBuf.DataHold.inthdr.hdroptions1 & RF_DEVICE_HEADER_OPTION1_BROADCAST)	{
								//	For broadcast frames use the broadcast frame sequence number   
								if (!(pdst->TxBuf.DataHold.inthdr.hdroptions1 & RF_DEVICE_HEADER_OPTION1_BROADCAST_RPT))	{
									BDR.txsequence++;	
								}
								pdst->TxBuf.Transmit.payld.hdr.sequence = BDR.txsequence;
							}	else	{
								//	specifically addressed frame 
								pdst->txsequence += 1;
							}
						}

						if (pdst->TxBuf.DataHold.segmentno < 1)		{	pdst->TxBuf.DataHold.segmentno = 1;		}
						if (pdst->TxBuf.DataHold.totalsegment < 1)	{	pdst->TxBuf.DataHold.totalsegment = 1;	}

						//	Build packet header (each packet has its own header)
						pktheader.length  		= len+sizeof(TYPE_EXTERNAL_RF_PACKET_HEADER)-1;
						pktheader.type	 		= pdst->TxBuf.DataHold.inthdr.subtype;
						pktheader.totalsegments	= pdst->TxBuf.DataHold.totalsegment;
						pktheader.segmentno		= pdst->TxBuf.DataHold.segmentno;
#ifdef	RF_NETWORK_MASTER_DEVICE
						pktheader.options = 0;
						if (RFAPP_IsControlSystemOnLine())	{
							pktheader.options |= RF_HEADER_EXPANSION_CS_CONNECT_STATE;	
							if			(RFAPP_IsCIPOnLine())	{
								pktheader.options |= RF_HEADER_EXPANSION_ETHERNET_NET;
							}	else if (RFAPP_IsCresenetOnLine())	{
								pktheader.options |= RF_HEADER_EXPANSION_CRESNET_NET;
							}	
						}
						pktheader.options	   |= RF_DEVICE_TYPE_GATEWAY;
#else
						pktheader.options		= 0;
#endif

						//	Copy packet header to transmit buffer
						memcpy (&pdst->TxBuf.Transmit.payld.data[pdst->TxBuf.Transmit.len], (UINT8 *) &pktheader, sizeof(TYPE_EXTERNAL_RF_PACKET_HEADER));
						pdst->TxBuf.Transmit.len += sizeof(TYPE_EXTERNAL_RF_PACKET_HEADER);
						//	copy packet data from Data Hold buffer to transmit buffer
						memcpy ((INT8 *) &pdst->TxBuf.Transmit.payld.data[pdst->TxBuf.Transmit.len], (INT8 *) &pdst->TxBuf.DataHold.data[pdst->TxBuf.DataHold.index], len);
						pdst->TxBuf.Transmit.len   += len;
						pdst->TxBuf.DataHold.len   -= len;
						pdst->TxBuf.DataHold.index += len;

						//	(6/5/07) save call back information for each packet and index to the start address of the packet  
						memcpy ((UINT8*)&pdst->TxBuf.Transmit.txcb[cbcnt], (UINT8*)&pdst->TxBuf.DataHold.inthdr.cb, sizeof(pdst->TxBuf.Transmit.txcb[0]));
						pdst->TxBuf.Transmit.txcb[cbcnt].index = pdst->TxBuf.Transmit.len-(len+sizeof(TYPE_EXTERNAL_RF_PACKET_HEADER));	
						cbcnt += 1;
						
						//	update concatenated packet count
						pdst->COMLog.txpacketconcatcount += 1;

						//	If all segments were sent clear parameters
						if (pdst->TxBuf.DataHold.totalsegment > 1)	{
							if (pdst->TxBuf.DataHold.segmentno++ >= pdst->TxBuf.DataHold.totalsegment)	{
								pdst->TxBuf.DataHold.segmentno    = 1;
								pdst->TxBuf.DataHold.totalsegment = 1;
							}
						}

						if (pdst->TxBuf.DataHold.inthdr.hdroptions1 & RF_DEVICE_HEADER_OPTION1_DONOT_CONCATE)	{
							//	no concatenation for this packet (concatenation is disabled for broadcast repeat packets, 
							//	this is to make sure multiple rpts are not sent in one tx attempt) 
							loop = FALSE;
						}
					}
					break;

				case	RF_PACKET_TYPE_MLM_REQUEST:
				case	RF_PACKET_TYPE_ASP_REQUEST:
					//	Not allowed to mix the MCP and MLM/ASP packets together
					if (pdst->TxBuf.Transmit.len == 0)	{
						//	Copy the internal header
						memcpy ((INT8 *) &pdst->TxBuf.Transmit.inthdr, (INT8 *) &pdst->TxBuf.DataHold.inthdr, sizeof(pdst->TxBuf.DataHold.inthdr));
						//	MLM/ASP packets, copy MLM or ASP data (for the MLM and ASP packets there is no payload header)
						//	The data is copied starting at the payload header section
						memcpy ((INT8*) &pdst->TxBuf.Transmit.payld.hdr, (INT8*) &pdst->TxBuf.DataHold.data, pdst->TxBuf.DataHold.len);
						pdst->TxBuf.Transmit.len   = pdst->TxBuf.DataHold.len;
						//	clear message length
						pdst->TxBuf.DataHold.len   = 0;
						pdst->TxBuf.DataHold.index = 0;

						//	(6/5/07) save call back information for each packet
						memcpy ((UINT8*)&pdst->TxBuf.Transmit.txcb[cbcnt], (UINT8*)&pdst->TxBuf.DataHold.inthdr.cb, sizeof(pdst->TxBuf.Transmit.txcb[0]));
						cbcnt += 1;
					}
					loop = FALSE;
					break;

				default:
					//	Unknown packet type, error condition

					//	Log error
					RFIntHostApp.pF_ErrorPrintf ("RFTransmit could not build packet (Unknown packet type=%02x, index=%02x)", pdst->TxBuf.DataHold.inthdr.type, index);

					pdst->TxBuf.DataHold.len   = 0;
					pdst->TxBuf.DataHold.index = 0;
					break;
			}
		}	else	{
			//	Try to read data from the pipe into the data hold buffer
#if defined (OS_NUCLEUS)
			if (NU_Receive_From_Pipe(&pdst->pipe, (void *) &(pdst->TxBuf.DataHold.inthdr), sizeof(TYPE_RF_DEVICE_RF_DATA_HOLD_BUF), (UINT32*) &hldlen, NU_NO_SUSPEND) == NU_SUCCESS)	{
#elif defined (OS_WINCE)
			//if (ReadMsgQueue(pdst->QueueReadHandle, (void *) &(pdst->TxBuf.DataHold.inthdr), sizeof (TYPE_RF_DEVICE_RFTX_INTERNAL_HDR), (UINT32*) &hldlen, OS_NO_SUSPEND, &dwFlags) == TRUE) {
			if (ReadMsgQueue(pdst->QueueReadHandle, (void *) &(pdst->TxBuf.DataHold), sizeof(TYPE_RF_DEVICE_RF_DATA_HOLD_BUF), (UINT32*) &hldlen, OS_NO_SUSPEND, &dwFlags) == TRUE) {
#endif
			hldlen = OsDequeueBytesWait(pdst->QueueHandle, (void*) &(pdst->TxBuf.DataHold), sizeof(TYPE_RF_DEVICE_RF_DATA_HOLD_BUF), OS_NO_WAIT);
			HPRF_QUEUE_DEBUG(DmConsolePrintf("*** %s:%i <--- dequeued %li bytes from desc[%i]\n", \
		            __FUNCTION__, __LINE__, hldlen, index);)
			if( hldlen > 0 ) {
				//	HS 11/07/07 
				//	check if extracted message is already expired due to result code timeouts or other PHY issues
				//	expired messages are discarded and the caller is informed
				if (SUCCESS != RFTransmit_CheckifMessagehastimetolive (pdst->TxBuf.DataHold.inthdr.cb.timetolivestart, pdst->TxBuf.DataHold.inthdr.cb.timetolive))	{
					if (pdst->TxBuf.DataHold.inthdr.cb.pf)	{
						pdst->TxBuf.DataHold.inthdr.cb.pf(pdst->TxBuf.DataHold.inthdr.cb.par, gTransactionExpired_c);
					}
					pdst->TxBuf.DataHold.len = 0;
				}	else	{
					//	adjust frame length, remove the header size from data field
					pdst->TxBuf.DataHold.len  = hldlen - sizeof(TYPE_RF_DEVICE_RFTX_INTERNAL_HDR);
				}
			}	else	{
				//	No more data in the pipe, exit the loop
				loop = FALSE;
			}
			//	Clear MCP packet index and pointers
			pdst->TxBuf.DataHold.index 		  = 0;
			pdst->TxBuf.DataHold.segmentno    = 1;
			pdst->TxBuf.DataHold.totalsegment = 1;
		}
	}	while (loop);

	return (pdst->TxBuf.Transmit.len);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function processes the individual device
 *
 * \detail
 * \date        08/21/2006
 * \param	    devices's no
 *
 * \return		None
 *
 * \retval
 */
static void RFTransmit_ProcessIndividualDeviceEvent	(UINT8 index)	{
	UINT8	buildAndPostpkt = FALSE;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = (g_pRFDeviceDescriptiveTable+index);
	//UINT16  res;

	HPRF_DEBUG(DmConsolePrintf("========== %s ========== index = %i waitingForAck = %i\n", \
	        __FUNCTION__, index, pdst->waitingforACK);)
	//	Check if device is waiting for an ack
	if (pdst->waitingforACK)	{
		if (pdst->resultcodeavailable)	{
			//	process result code
			switch (pdst->resultcode)	{
				case gSuccess_c:
					//	packet transmission was successful
					pdst->waitingforACK = FALSE;
					pdst->resultcodeavailable = FALSE;
					pdst->txpacketretrycntr = 0;
					buildAndPostpkt = TRUE;
					break;

				case	gNoAck_c:
				case	gTransactionOverflow_c:
				case	gTransactionExpired_c:
				case	gChannelAccessFailure_c:
					//	The retry check is only needed here for the ASP and MLME packets, MCP packets that failed after maximum retries are
					//	handled in the retry processing function in the rf receive modules
					//	If the retry count is zero MCP packet which receives NO_ACK would not report the NO_ACK result code here
					if (((RFDesc_GetMLMEDescriptiveTableIndex()==index) || (RFDesc_GetASPDescriptiveTableIndex()==index)) && (pdst->txpacketretrycntr >= pdst->maxtxpacketretries))	{	
						//	All attempts failed for ASP and MLME packets, allow the next ASP or MLME packet to be sent
						buildAndPostpkt = TRUE;
					}	else	{
						//	retransmit MCP packet
						pdst->txpacketretrycntr += 1;
						RFTransmit_PostPacketToRFStack (index);
					}
					break;

				case	gInvalidGts_c:
				case	gUnavailableKey_c:
				case	gInvalidParameter_c:
				case	gFrameTooLong_c:
				case	gFailedSecurityCheck_c:
				default:
					RFIntHostApp.pF_ErrorPrintf("Unexpected RF Stack result: status[%02x] index[%02x]\r\n", pdst->resultcode, index);
			#ifdef	RF_MSG_ERR_LOG
					RFComLog_PutRFDeviceMsgError (index, pdst->TxBuf.Transmit.inthdr.type, pdst->TxBuf.Transmit.inthdr.subtype,
													pdst->TxBuf.Transmit.inthdr.req, RF_COMERROR_LOG_INIT_CONFIRMFUNC, pdst->resultcode);
			#endif
					buildAndPostpkt = TRUE;
					break;
			}
		}
	}	else	{
		pdst->resultcodeavailable = FALSE;
		buildAndPostpkt = TRUE;
	}

	if (buildAndPostpkt)	{
		//	Build packet and post to transmit buffer
		//	Some ASP/MLME packets do not return any error code, therefore send other packets pending in the buffer
		pdst->waitingforACK = FALSE;
		while (pdst->waitingforACK == FALSE)	{
			if	(RFTransmit_BuildPacket(index))	{
				RFTransmit_PostPacketToRFStack (index);
			}	else	{
				break;
			}
		}
	}
}


/**
 *
 * \author      Hazrat Shah
 *
 * \brief		The RF Transmit task
 *
 * \detail		This task waits on individual RF Device "Trigger RF Trasmit Task" events. If the event code is set it then
 *				processes the device's result code if one is pending and checks the device's pipe for messages to the device.
 *
 * \date        08/21/2006
 *
 * \param	    None
 *
 * \return		None
 *
 * \retval
 */
#if defined (OS_NUCLEUS)
static void RFTransmit_Task (UINT32 argc, void * argv)	{
#elif defined (OS_WINCE)
static DWORD RFTransmit_Task (void * pContext)	{
#endif
static void RFTransmit_Task(UINT32 param) {
#if defined (OS_NUCLEUS)
	UINT32 TriggerRFTransmitTaskMainEvents, TriggerRFTransmitTaskSubEvents;
	UINT32 i, j;
#elif defined (OS_WINCE)
	//DWORD dwResult;
	UINT32 Events[RFTRANS_EVENTS_ARRAY_COUNT], i;
#endif
	UINT32 i, j;
	UINT32 TriggerRFTransmitTaskMainEvents, TriggerRFTransmitTaskSubEvents;

	RFIntHostApp.pF_StartupWait(HPRF_STARTUP_TRANS_READY);

	//	Wait for RF Stack to be intialized
	while (!RFReceive_IsRFStackInitialized())	{	RFIntHostApp.pF_HwDelayMsec(100);	}

#if defined (OS_WINCE)
	RFIntHostApp.pF_ErrorPrintf("\r\nRFTransmit_Task: Running!");
#endif

#if defined (OS_NUCLEUS)
    while(1)	{
		if (NU_Retrieve_Events(&RFTransTask.TriggerRFTransmitTaskMainEvents, 0xffffffff, NU_OR_CONSUME, &TriggerRFTransmitTaskMainEvents, NU_SUSPEND) == NU_SUCCESS)	{
			for (i=0; i<RFTRANS_TRIGGER_RF_TRANSMIT_TASK_MAINEVENTS; i++)	{
				if (TriggerRFTransmitTaskMainEvents & (1<<i))	{
					if (NU_Retrieve_Events(&RFTransTask.TriggerRFTransmitTaskSubEvents[i], 0xffffffff, NU_OR_CONSUME, &TriggerRFTransmitTaskSubEvents, NU_NO_SUSPEND) == NU_SUCCESS)	{
						for (j=0; j<RFTRANS_TRIGGER_RF_TRANSMIT_TASK_SUBEVENTS; j++)	{
							if (TriggerRFTransmitTaskSubEvents & (1<<j))	{
				  				RFTransmit_ProcessIndividualDeviceEvent(i*RFTRANS_TRIGGER_RF_TRANSMIT_TASK_SUBEVENTS+j);
							}
						}
					}
				}
			}
		}
	}
#elif defined (OS_WINCE)
	while (1)
	{
		// Wait for Multiple Objects (WinCE can not wait for more than 64 events (MAXIMUM_WAIT_OBJECTS)
		//dwResult = WaitForMultipleObjects(RFTRANS_TRIGGER_RF_TRANSMIT_EVENT_COUNT, RFTransTask.hEvents, FALSE, OS_SUSPEND);
		if (WAIT_OBJECT_0 == WaitForSingleObject(RFTransTask.hRfTransmitEvent, OS_SUSPEND))
		{
			// copy the global events to local storage and clear the global flags
			EnterCriticalSection(&RFTransTask.RfTransEventCriticalSection);
			memcpy(Events, RFTransTask.RfTransmitEventGroup, sizeof (RFTransTask.RfTransmitEventGroup));
			memset(RFTransTask.RfTransmitEventGroup, 0, sizeof (RFTransTask.RfTransmitEventGroup));
			LeaveCriticalSection(&RFTransTask.RfTransEventCriticalSection);
			
			// Go through the event flags to see which one is set and then process
			for (i = 0; i < RFTRANS_TRIGGER_RF_TRANSMIT_EVENT_COUNT; i++)
			{
				if (Events[i/32] & (1 << (i % 32)))
				{
					// Pass the index
					RFTransmit_ProcessIndividualDeviceEvent((UINT8)i);
				}
			}
		}
		else
		{
			RFIntHostApp.pF_ErrorPrintf("RFTRXTASK: Failed to retrieve events: %d", GetLastError());
		}
	}
#endif
    while(1)    {
        TriggerRFTransmitTaskMainEvents = OsWaitForEventAndClear(RFTransTask.TriggerRFTransmitTaskMainEvents, 0xffffffff);
        for(i=0; i < RFTRANS_TRIGGER_RF_TRANSMIT_TASK_MAINEVENTS; ++i) {
            if(TriggerRFTransmitTaskMainEvents & (1<<i)) {
                TriggerRFTransmitTaskSubEvents = OsWaitEventGroupTimeout(RFTransTask.TriggerRFTransmitTaskSubEvents[i], 0xffffffff, OS_NO_WAIT);
                for(j=0; j < RFTRANS_TRIGGER_RF_TRANSMIT_TASK_SUBEVENTS; ++j) {
                    if(TriggerRFTransmitTaskSubEvents & (1<<j)) {
                        RFTransmit_ProcessIndividualDeviceEvent(i*RFTRANS_TRIGGER_RF_TRANSMIT_TASK_SUBEVENTS+j);
                    }
                }
            }
        }
    }
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function returns pipe avaliable memory size
 *
 * \detail
 * \date        09/06/2006
 *
 * \param	    UINT8 index
 *
 * \return		UINT32
 *
 * \retval		returns pipe avaliable memory size
 */
UINT32	RFTransmit_GetPipeAvailableMemorySizeByIndex (UINT8 index, UNSIGNED *ppipesize, UNSIGNED *pavailable, UNSIGNED *pmessages)	{
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = (g_pRFDeviceDescriptiveTable+index);
	INT32 tmpQueueSize = 0;
#if defined (OS_NUCLEUS)
	CHAR pipe_name[8];
	VOID *start_address;
	OPTION message_type;
	UNSIGNED message_size;
	OPTION suspend_type;
	UNSIGNED tasks_suspended;
	NU_TASK *first_task;

	//	Wait for pipe to be empty before posting a low priority message
	return (NU_Pipe_Information(&pdst->pipe, pipe_name, &start_address,
									ppipesize, pavailable, pmessages,
									&message_type, &message_size,
									&suspend_type, &tasks_suspended,
									&first_task));
#elif defined (OS_WINCE)
	MSGQUEUEINFO sMsgQueueInfo;
	sMsgQueueInfo.dwSize = sizeof (MSGQUEUEINFO);

	// Note that, the "pmessages" is the only variable which is used after this function returns, as far as
	// the system behaviour is concerned, the other 2 are used to print during debug.
	// Since we create queue with no limit on the messages, we fake the pipe size as 1 mega bytes, in case of 
	// Windows CE. In case we create the queue with limit on the message, we calculate the bytes in the queue
	// by multiplying the max message supported by queue with no. of bytes in each message.

	if (GetMsgQueueInfo (pdst->QueueReadHandle, &sMsgQueueInfo) == TRUE)
	{
		// If dwMaxMessages is zero, then there is no limit on the no of messages, so return some huge number, else
		// return the max no of message supported by queue multiplied by byes per message (total bytes in queue)
		*ppipesize = (UNSIGNED) (sMsgQueueInfo.dwMaxMessages ? (sMsgQueueInfo.dwMaxMessages * sMsgQueueInfo.cbMaxMessage) : (1024 * 1024));
		// No of message currently in queue
		*pmessages = (UNSIGNED)sMsgQueueInfo.dwCurrentMessages;
		// No of available bytes in queue
		*pavailable = (UNSIGNED) (*ppipesize - (*pmessages * sMsgQueueInfo.cbMaxMessage));

		return SUCCESS;
	}
	else
	{
		RFIntHostApp.pF_ErrorPrintf("RFTxGetPipeAvaliableMemorySize Fail: %d", GetLastError());
		return FAILURE;
	}

#endif
	tmpQueueSize = OsQueueSize(pdst->QueueHandle);
	if(tmpQueueSize)
	{
	    *ppipesize = tmpQueueSize;
	    *pmessages = OsGetQueueCount(pdst->QueueHandle);
	    *pavailable = (*ppipesize - *pmessages);
	    return SUCCESS;
	}
	return FAILURE;
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function posts the frame to rf transmit high priority pipe
 *
 * \detail
 * \date        09/06/2006
 *
 * \param	    UINT8 index, UINT8 *pbuf, UINT16 buflen, UINT8 Priority
 *
 * \return		UINT16
 *
 * \retval		returns SUCCESS if buffer was posted successfully, otherwise returns FAILURE
 */
static UINT16	RFTransmit_PostMessageToPipeByIndex (UINT8 index, UINT8 *pbuf, UINT16 buflen, UINT8 Priority)	{
	UINT16 result = FAILURE;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = (g_pRFDeviceDescriptiveTable+index);
	BOOL isTraceEnabled;
	UNSIGNED pipesize;
	UNSIGNED avaliable;
	UNSIGNED messages;
	STATUS   status = SUCCESS;
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF *pDataHold = (TYPE_RF_DEVICE_RF_DATA_HOLD_BUF*) pbuf;
	INT32 bytesQueued;
	UINT32	 i, maxwait;

	HPRF_DEBUG(DmConsolePrintf("========== %s ========== index = %i len=%hu\n", __FUNCTION__, index, buflen);)
	HPRF_DEBUG(dump_stack();)
	RFIntHostApp.pF_IsRFTxDebugTraceEnabled(&isTraceEnabled);

	if ((index <= g_MaxDescriptiveTableEntries) && g_MaxDescriptiveTableEntries) {

		//	Note: currently the low priority and high priority shares the same pipe
		if (Priority == RF_PACKET_PRIORITY_LOW)	{

			if (LowPriorityMessagePostingDelay)	{
				HwDelayMsec(LowPriorityMessagePostingDelay);
				LowPriorityMessagePostingDelay /= 2;
			}

			maxwait = 50;
			do	{
				//	Wait for all devices pipe to be empty before posting a low priority message
				for (i=0, messages=0; (i<g_MaxSupportedRFDevices) && (!messages); i++)	{					
					if (g_pRFDeviceDescriptiveTable[i].linkstate == RF_DEVICE_LINK_STATE_ACTIVE)	{
						status = RFTransmit_GetPipeAvailableMemorySizeByIndex (i, &pipesize, &avaliable, &messages);
						if (SUCCESS != status) {
							RFIntHostApp.pF_ErrorPrintf("RFTRANS: transmit pipe information read error for descriptive entry[%02x]\r\n", i);							
							break;
						}	
					}
				}
				if (messages)	{	RFIntHostApp.pF_HwDelayMsec (10);	}
			} while ((status == SUCCESS) && messages && --maxwait);
		}

		if (buflen <= RF_DEVICE_PIPE_MAX_MSG_LEN)	{							
#if defined (OS_NUCLEUS)
			if (NU_SUCCESS != NU_Send_To_Pipe(&pdst->pipe, (INT8*) pbuf, buflen, (RFTRANS_PIPE_TIMEOUT)))	{
#elif defined (OS_WINCE)
			if (TRUE != WriteMsgQueue(pdst->QueueWriteHandle, (VOID *)pbuf, buflen, RFTRANS_PIPE_TIMEOUT, 0)) {
#endif
			HPRF_QUEUE_DEBUG(DmConsolePrintf("--- %s:%i ---> queueing %lu bytes to desc[%i]\n", \
                    __FUNCTION__, __LINE__, buflen, index);)
			bytesQueued = OsQueueBytes(pdst->QueueHandle, (void*)pbuf, buflen);
			if(bytesQueued <= 0) {
				g_pRFDeviceDescriptiveTable[index].COMLog.txbufoverflowcount++;
				//	Error, message could not be posted to the PIPE?
				RFIntHostApp.pF_ErrorPrintf("RFTRANS: descriptive entry[%02x] transmit pipe full, msg type[%x] stype[%x] priority[%x] len[%x] reseting tx pipe, \r\n", index, pDataHold->inthdr.type, pDataHold->inthdr.subtype, Priority, buflen);
				//	reseting pipe
				RFTransmit_ResetTxBuffers (index);
				//	inform host application about the transmit pipe overflow condition
				RFIntHostApp.pF_TransmitPipeOverFlowOccured(index);
			}
			else {
			    HPRF_QUEUE_DEBUG(DmConsolePrintf("%s - index[%hu] queued %li bytes\n", __FUNCTION__, index, bytesQueued);)
				//	raise the pipe event flags
				RFTransmit_TriggerRFTransmitTaskEvent (index);
				result = SUCCESS;
			}
		}
		else {
			//	Error, message could not be posted to the PIPE?
			RFIntHostApp.pF_ErrorPrintf("RFTRANS: packet is larger than allocated pipe msg size[%04x] descriptive entry[%02x]\r\n", buflen, index);
		}
	}

	return (result);
}

////////////////////////////////////////////////////////////////////////////////
//	GLOBAL FUNCTIONS
//////////////////////////////////////////////////////////////////////////////

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function is called on startup to create the rf transmit task, events etc..
 *
 * \detail		The RF transmit task, event flags are created, .... are created
 *
 * \date        08/22/2006
 * \param	    None
 *
 * \return		None
 *
 * \retval
 */
void	RFTransmit_Init (void)	{

#if defined (OS_NUCLEUS)	
	UINT16	i;
#endif
	UINT16 i;
	BOOL	uidvalid;

	RFIntHostApp.pF_IsUIDValid(&uidvalid);

	if (uidvalid)	{
#if defined (OS_NUCLEUS)
		//	Create trigger rf transmit main event group
		if	(NU_Create_Event_Group(&RFTransTask.TriggerRFTransmitTaskMainEvents, "ERFTXMAN") != NU_SUCCESS)	{
    		RFIntHostApp.pF_ErrorHandler("ERFTXMAN: Could not create trigger rf transmit main event group!");
		}

		//	Create trigger rf transmit sub event group
		for (i=0; i<RFTRANS_TRIGGER_RF_TRANSMIT_TASK_MAINEVENTS; i++)	{
			if	(NU_Create_Event_Group(&RFTransTask.TriggerRFTransmitTaskSubEvents[i], "ERFTXSUB") != NU_SUCCESS)	{
    			RFIntHostApp.pF_ErrorHandler("ERFTXSUB: Could not create trigger rf transmit sub-event group!");
    		}
		}

		//	Create rf stack buffer available event group
		if	(NU_Create_Event_Group(&RFTransTask.rfstackbufferavailableevents, "ERFTXBA") != NU_SUCCESS)	{
    		RFIntHostApp.pF_ErrorHandler("ERFTXBA: Could not create tranmission buffer available event group!");
		}

		//	Create RF trasmit task
		if(NU_Create_Task(&RFTransTask.task, "TRFTRANS", RFTransmit_Task, 0, NU_NULL,
			RFTransTask.stack, sizeof(RFTransTask.stack), RFTRANS_TASK_PRIORITY, 0, NU_PREEMPT, NU_START ) != NU_SUCCESS)	{
       		RFIntHostApp.pF_ErrorHandler("TRFTRANS: Could not create task!");
		}
#elif defined (OS_WINCE)
		// Initialize the critical section
		InitializeCriticalSection(&RFTransTask.RfTransEventCriticalSection);

		//	Create trigger rf transmit main event group
		//for (i = 0; i < RFTRANS_TRIGGER_RF_TRANSMIT_EVENT_COUNT; i++)
		//{
		//	// Create so many events
		//	RFTransTask.hEvents[i] = CreateEvent(NULL, FALSE, FALSE, NULL);
		//	if (RFTransTask.hEvents[i] == NULL)
		//	{
		//		RFIntHostApp.pF_ErrorHandler("RFTransmit_Init: Could not create events: %d!", i);
		//	}
		//}

		// Clear all the events (to be safe)
		memset (RFTransTask.RfTransmitEventGroup, 0, sizeof (RFTransTask.RfTransmitEventGroup));

		// Create main event
		RFTransTask.hRfTransmitEvent = CreateEvent(NULL, FALSE, FALSE, NULL);
		if (RFTransTask.hRfTransmitEvent == NULL)
		{
			RFIntHostApp.pF_ErrorPrintf("RFTransmit_Init: Could not create RF transmit event");
		}

		//	Create rf stack buffer available event group
		RFTransTask.hRfStackbufferavailableevents = CreateEvent(NULL, FALSE, FALSE, NULL);
		if (RFTransTask.hRfStackbufferavailableevents  == NULL)
		{
			RFIntHostApp.pF_ErrorPrintf("RFTransmit_Init: Could not create tranmission buffer available event group!");
		}

		//	Create RF trasmit task
		RFTransTask.task = CreateThread(NULL, 0, RFTransmit_Task, pHPRDevice, 0, NULL);
		if (RFTransTask.task == NULL)
		{
			RFIntHostApp.pF_ErrorPrintf("TRFTRANS: Could not create task!");
		}
		// Set thread priority
		CeSetThreadPriority(RFTransTask.task, RFTRANS_TASK_PRIORITY);
#endif
        //  Create trigger rf transmit main event group
		RFTransTask.TriggerRFTransmitTaskMainEvents = OsCreateEventGroup();
		if(!RFTransTask.TriggerRFTransmitTaskMainEvents) {
		    DmConsolePrintf("ERFTXMAN: Could not create trigger RF transmit main event group!");
		}
		for(i=0; i < RFTRANS_TRIGGER_RF_TRANSMIT_TASK_MAINEVENTS; ++i) {
		    RFTransTask.TriggerRFTransmitTaskSubEvents[i] = OsCreateEventGroup();
		    if(!RFTransTask.TriggerRFTransmitTaskSubEvents[i]) {
		        DmConsolePrintf("ERFTXSUB: Could not create trigger RF transmit sub-event group!");
		    }
		}
		if(!OsCreateEvent(&RFTransTask.hRfStackBufferAvailableEvents)) {
		    DmConsolePrintf("ERFTXBA: Could not create tranmission buffer available event group!");
		}
		RFTransTask.task = OsCreateNamedAdvTaskPri(RFTransmit_Task, 0, 0, RFTRANS_TASK_PRIORITY, "TRFTRANS");
        if(!RFTransTask.task) {
            DmConsolePrintf("TRFTRANS: Could not create task!");
        }
	}
}

/////////////////////////////////////////////////////////////////////////////////////////
//	Building MCP data packets
/////////////////////////////////////////////////////////////////////////////////////////
/**
 *
 * \author      Hazrat Shah
 *
 * \brief		Post message to transmit pipe by RFID
 *
 * \detail
 * \date        08/28/2006
 * \param	    UINT8 rfid, UINT8 *pbuf, UINT16 buflen, UINT8 priority
 *
 * \return		UINT16
 *
 * \retval		returns 0 if buffer was posted successfully, otherwise returns buflen
 */
UINT16	RFTransmit_PostMessageToPipeByRFID (UINT8 rfid, UINT8 *pbuf, UINT16 buflen, UINT8 priority)	{
	UINT8 	index[MAX_DESCRIPTIVE_TABLE_ENTRIES+2];
	UINT16 	indexcnt, i;
	UINT16  res = FAILURE;

	//	lookup rfid indexs in the descriptive table
	if ((indexcnt = RFDesc_GetActiveRFDeviceTableIndexByRFID(rfid, index)))	{
		res = SUCCESS;
		//	Post packets to all device(s) pipes, multiple RFID devices entries may occur at setup time only
		//	In PNP setup mode all devices may be set to RFID 1
		for	(i=0; i<indexcnt;  i++)	{
			if (RFTransmit_PostMessageToPipeByIndex (index[i], pbuf, buflen, priority) != SUCCESS){	res = FAILURE;	}
		}
	}

	return ((res == SUCCESS)?0:buflen);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		Build and Post RF Device packet use device's RFID to find destination UID address
 *
 * \detail
 * \date        08/28/2006
 * \param	    UINT8 rfid, UINT8 *pbuf, UINT16 buflen, INT32(*pcbf)(UINT32 par, UINT32 status), UINT32 cbfpar, UINT32 timetolive
 *
 * \return		buflen
 *
 * \retval		returns 0 if buffer was posted successfully, otherwise returns buflen
 */
UINT16 RFTransmit_BuildCresnetPacketAndPostToPipeByRFID (UINT8 rfid, UINT8 *pbuf, UINT16 len, INT32(*pcbf)(UINT32 par, UINT32 status), UINT32 cbfpar, UINT32 timetolive)	{
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	  = RF_PACKET_TYPE_MCP_REQUEST;
	DataHold.inthdr.subtype 	  = RF_PACKET_CRESNET;
	DataHold.inthdr.hdroptions1   = (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);
	DataHold.inthdr.cb.pf		  =	pcbf;
	DataHold.inthdr.cb.par		  = cbfpar;
	DataHold.inthdr.cb.timetolivestart = OS_RetrieveClock();
	DataHold.inthdr.cb.timetolive = timetolive;
	
	memcpy ((INT8 *) &DataHold.data, pbuf, len);

	return (RFTransmit_PostMessageToPipeByRFID (rfid, (UINT8 *) &DataHold, sizeof(DataHold.inthdr)+len, RF_PACKET_PRIORITY_HIGH));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		Build and Post RF Device packet use device's descriptive table entry to lookup destination UID address
 *
 * \detail
 * \date        06/05/2007
 *
 * \param	    UINT8 index, UINT8 *pbuf, UINT16 len, INT32(*pcbf)(UINT32 par, UINT32 status), UINT32 cbfpar, UINT32 timetolive
 *
 * \return		buflen
 *
 * \retval		returns 0 if buffer was posted successfully, otherwise returns buflen
 */
UINT16 RFTransmit_BuildCresnetPacketAndPostToPipeByIndex (UINT8 index, UINT8 *pbuf, UINT16 len, INT32(*pcbf)(UINT32 par, UINT32 status), UINT32 cbfpar, UINT32 timetolive)	{
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	 = RF_PACKET_TYPE_MCP_REQUEST;
	DataHold.inthdr.subtype 	 = RF_PACKET_CRESNET;
	DataHold.inthdr.hdroptions1  = (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);
	DataHold.inthdr.cb.pf		 =	pcbf;
	DataHold.inthdr.cb.par		 = cbfpar;
	DataHold.inthdr.cb.timetolivestart = OS_RetrieveClock();
	DataHold.inthdr.cb.timetolive = timetolive;
	
	memcpy ((INT8 *) &DataHold.data, pbuf, len);
	return (RFTransmit_PostMessageToPipeByIndex(index, (UINT8 *) &DataHold, sizeof(DataHold.inthdr)+len, RF_PACKET_PRIORITY_HIGH));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		Build and Post Local cresnet packets use device's descriptive table entry to lookup destination UID address
 *
 * \detail
 * \date        06/05/2007
 *
 * \param	    UINT8 index, UINT8 *pbuf, UINT16 len, INT32(*pcbf)(UINT32 par, UINT32 status), UINT32 cbfpar, UINT32 timetolive
 *
 * \return		UINT16
 *
 * \retval
 */
UINT16 RFTransmit_BuildLocalCresnetPacketAndPostToPipeByIndex (UINT8 index, UINT8 *pbuf, UINT16 len, INT32(*pcbf)(UINT32 par, UINT32 status), UINT32 cbfpar, UINT32 timetolive)	{
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	  = RF_PACKET_TYPE_MCP_REQUEST;
	DataHold.inthdr.subtype 	  = RF_PACKET_LOCAL_CRESNET;
	DataHold.inthdr.hdroptions1	  = (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);
	DataHold.inthdr.cb.pf		  =	pcbf;
	DataHold.inthdr.cb.par		  = cbfpar;
	DataHold.inthdr.cb.timetolivestart = OS_RetrieveClock();
	DataHold.inthdr.cb.timetolive = timetolive;

	memcpy ((INT8 *) &DataHold.data, pbuf, len);
	return (RFTransmit_PostMessageToPipeByIndex(index, (UINT8 *) &DataHold, sizeof(DataHold.inthdr)+len, RF_PACKET_PRIORITY_HIGH));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		Build and Post Throughput packets use device's descriptive table entry to lookup
 *				destination UID address
 *
 * \detail
 * \date        06/19/2007
 *
 * \param	    UINT8 index, UINT8 *pbuf, UINT8 len, TYPE_RFSTACK_CB *pcb, UINT8 Broadcast, UINT8 nocat
 *
 * \return		UINT16
 *
 * \retval
 */
UINT16 RFTransmit_BuildThroughPutPacketAndPostToPipeByIndex (UINT8 index, UINT8 *pbuf, UINT16 len, TYPE_RFSTACK_CB *pcb, UINT8 Broadcast, UINT8 nocat)	{
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;
	UINT16 res = FAILURE;

	if (len < RF_DEVICE_DATA_HOLD_BUFFER_SIZE)	{
		//	Clear internal header
		memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));
		//	Build rf device internal packet header
		DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST;
		DataHold.inthdr.subtype 	= RF_PACKET_THROUGHPUT_TST;
		
		DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);
		if (Broadcast)	{
			DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_BROADCAST|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);
		}

		if (nocat)	{
			DataHold.inthdr.hdroptions1	|=	RF_DEVICE_HEADER_OPTION1_DONOT_CONCATE; 
		}

		DataHold.inthdr.cb.pf		= pcb->pf;
		DataHold.inthdr.cb.par		= pcb->par;

		memcpy ((INT8 *) &DataHold.data, pbuf, len);
		res = RFTransmit_PostMessageToPipeByIndex(index, (UINT8 *) &DataHold, sizeof(DataHold.inthdr)+len, RF_PACKET_PRIORITY_HIGH);
	}

	return (res);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		Build and Post RF Device packet use device's RFID to find destination UID address
 *				caller is informed with the result of rf transmission (ACK/NAK)
 *
 * \detail
 * \date        06/05/2007
 *
 * \param	    UINT8 rfid, UINT8 *pbuf, UINT16 len, UINT8 priority, INT32(*pcbf)(UINT32 par, UINT32 status), UINT32 cbfpar, UINT32 timetolive
 *
 * \return		buflen
 *
 * \retval		returns 0 if buffer was posted successfully, otherwise returns buflen
 */
UINT16 RFTransmit_BuildLocalCresnetPacketAndPostToPipeByRFID (UINT8 rfid, UINT8 *pbuf, UINT16 len, UINT8 priority, INT32(*pcbf)(UINT32 par, UINT32 status), UINT32 cbfpar, UINT32 timetolive)	{
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST;
	DataHold.inthdr.subtype 	= RF_PACKET_LOCAL_CRESNET;
	DataHold.inthdr.hdroptions1 = (RF_DEVICE_HEADER_OPTION1_REQ_ACK)|(priority?RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH:0);
	DataHold.inthdr.cb.pf		  =	pcbf;
	DataHold.inthdr.cb.par		  = cbfpar;
	DataHold.inthdr.cb.timetolivestart = OS_RetrieveClock();
	DataHold.inthdr.cb.timetolive = timetolive;

	memcpy ((INT8 *) &DataHold.data, pbuf, len);

	return (RFTransmit_PostMessageToPipeByRFID (rfid, (UINT8 *) &DataHold, sizeof(DataHold.inthdr)+len, priority));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function is prepares and posts gateway startup message to devices
 *
 * \detail
 *
 * \date        06/06/2007
 *
 * \param	    UINT8 index, BOOL Broadcast, UINT32 flags
 *
 * \return		UINT16
 *
 * \retval		SUCCESS/FAILURE
 */
UINT16 RFTransmit_BuildGatewayReStartPacketAndPostToPipeByIndex (UINT8 index, BOOL broadcast, UINT32 flags, UINT8 repeats)	{
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;
	TYPE_RF_GATEWAY_RESTART_PACKET_DATA *GwRS = (TYPE_RF_GATEWAY_RESTART_PACKET_DATA*) &DataHold.data;
	UINT16	res;

	//	Clear internal header and data field
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));
	memset (GwRS, 0, sizeof(TYPE_RF_GATEWAY_RESTART_PACKET_DATA));

	//RFReceive_GetLocalDeviceUID (GwRS->uid);
	//GwRS->panid = RFReceive_GetLocalDevicePanID();
	GwRS->flags = flags;

	//	Build rf device internal packet header
	DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST;
	DataHold.inthdr.subtype 	= RF_PACKET_GATEWAY_RESTART;
	
	if (broadcast)	{
		DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH|RF_DEVICE_HEADER_OPTION1_BROADCAST);
	}	else	{
		DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);
	}

	if (repeats)	{
		DataHold.inthdr.hdroptions1	|= RF_DEVICE_HEADER_OPTION1_DONOT_CONCATE;
	}

	res = RFTransmit_PostMessageToPipeByIndex(index, (UINT8 *) &DataHold, sizeof(DataHold.inthdr)+sizeof(TYPE_RF_GATEWAY_RESTART_PACKET_DATA), RF_PACKET_PRIORITY_HIGH);
	while (repeats-- && (res == SUCCESS))	{
		if (broadcast)	{	DataHold.inthdr.hdroptions1	|= RF_DEVICE_HEADER_OPTION1_BROADCAST_RPT;	}
		res = RFTransmit_PostMessageToPipeByIndex(index, (UINT8 *) &DataHold, sizeof(DataHold.inthdr)+sizeof(TYPE_RF_GATEWAY_RESTART_PACKET_DATA), RF_PACKET_PRIORITY_HIGH);
	}

	return (res);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function posts the echo request packet to the passed on RFID
 *
 * \detail
 *
 * \date        06/06/2007
 *
 * \param	    UINT8 rfid, UINT8 *pBuf, UINT8 Len
 *
 * \return		UINT16
 *
 * \retval		SUCCESS/FAILURE
 */
UINT16 RFTransmit_BuildEchoRequestPacketAndPostToPipeByRFID (UINT8 rfid, UINT8 *pBuf, UINT8 Len)	{
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;
	UINT16	result = FAILURE;
	TYPE_RF_ECHO_REQUEST_PACKET_DATA *pER;
	UINT8 	indexbuf[MAX_DESCRIPTIVE_TABLE_ENTRIES+2];
	UINT16 	indexcnt = RFDesc_GetActiveRFDeviceTableIndexByRFID (rfid, indexbuf);

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	if (Len && indexcnt)	{
		//	Build rf device internal packet header
		DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST;
		DataHold.inthdr.subtype 	= RF_PACKET_ECHO_REQUEST;
		DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);

		pER = (TYPE_RF_ECHO_REQUEST_PACKET_DATA*) &DataHold.data;
		pER->linkquality = g_pRFDeviceDescriptiveTable[indexbuf[0]].LQ.linkqualityAtLocalDevice;
		memcpy ((INT8 *) &pER->data, (INT8*) pBuf, Len);
		result = (RFTransmit_PostMessageToPipeByIndex (indexbuf[0], (UINT8*) &DataHold, sizeof(DataHold.inthdr)+Len+sizeof(pER->linkquality), RF_PACKET_PRIORITY_HIGH));
	}

	return (result);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function posts the echo request packet to the device in the given descriptive entry
 *
 * \detail
 *
 * \date        06/06/2007
 *
 * \param	    UINT8 index, UINT8 *pBuf, UINT8 Len
 *
 * \return		UINT16
 *
 * \retval		SUCCESS/FAILURE
 */
UINT16 RFTransmit_BuildEchoRequestPacketAndPostToPipeByIndex (UINT8 index, UINT8 *pBuf, UINT8 Len)	{
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;
	UINT16	result = FAILURE;
	TYPE_RF_ECHO_REQUEST_PACKET_DATA *pER;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	if (Len && index)	{
		//	Build rf device internal packet header
		DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST;
		DataHold.inthdr.subtype 	= RF_PACKET_ECHO_REQUEST;
		DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);

		pER = (TYPE_RF_ECHO_REQUEST_PACKET_DATA*) &DataHold.data;
		pER->linkquality = g_pRFDeviceDescriptiveTable[index].LQ.linkqualityAtLocalDevice;
		memcpy ((INT8 *) &pER->data, (INT8*) pBuf, Len);
		result = (RFTransmit_PostMessageToPipeByIndex (index, (UINT8*) &DataHold, sizeof(DataHold.inthdr)+Len+sizeof(pER->linkquality), RF_PACKET_PRIORITY_HIGH));
	}

	return (result);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function posts the echo reply packet to the device in the given descriptive entry
 *
 * \detail
 *
 * \date        06/06/2007
 *
 * \param	    UINT8 index, UINT8 *pBuf, UINT8 Len
 *
 * \return		UINT16
 *
 * \retval		SUCCESS/FAILURE
 */
UINT16 RFTransmit_BuildEchoReplyPacketAndPostToPipeByIndex (UINT8 index, UINT8 *pEchoData, UINT8 Len)	{
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;
	TYPE_RF_ECHO_REPLY_PACKET_DATA  *pER;
	//UINT16 i;
	UINT16 result = FAILURE;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	if (Len)	{
		if (Len>RF_ECHO_PACKET_DATA_LEN)	{	Len = RF_ECHO_PACKET_DATA_LEN;	}
		//	Build rf device internal packet header
		DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST;
		DataHold.inthdr.subtype 	= RF_PACKET_ECHO_REPLY;
		DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);

		pER = (TYPE_RF_ECHO_REPLY_PACKET_DATA*) &DataHold.data;
		pER->linkquality = g_pRFDeviceDescriptiveTable[index].LQ.linkqualityAtLocalDevice;
		memcpy ((INT8*) &pER->data, (INT8*) pEchoData, Len);
		result = RFTransmit_PostMessageToPipeByIndex(index, (UINT8 *) &DataHold, sizeof(DataHold.inthdr)+Len+sizeof(pER->linkquality), RF_PACKET_PRIORITY_HIGH);
	}

	return (result);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function posts the link request packet to the device in the given descriptive entry
 *
 * \detail
 *
 * \date        06/06/2007
 *
 * \param	    UINT8 index, UINT8 flags, UINT8 linkcondition, UINT8 deviceoptions, UINT16 randomno, INT8 *pIDStr, BOOL broadcast, UINT8 repeats
 *
 * \return		UINT16
 *
 * \retval		SUCCESS/FAILURE
 */
UINT16 RFTransmit_BuildLinkRequestPacketAndPostToPipeByIndex (UINT8 index, UINT8 flags, UINT8 linkcondition, UINT8 deviceoptions, UINT16 randomno, INT8 *pIDStr, BOOL broadcast, UINT8 repeats)	{	
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF  DataHold;
	TYPE_RF_LINK_REQUEST_PACKET_DATA lreq;	
	UINT16	len, slen, power, res;

	//	Clear internal header
	memset ((UINT8*) &DataHold, 0, sizeof(TYPE_RF_DEVICE_RF_DATA_HOLD_BUF));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST;
	DataHold.inthdr.subtype 	= RF_PACKET_LINK_REQUEST;
	if (broadcast)	{
		DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH|RF_DEVICE_HEADER_OPTION1_BROADCAST);
	}	else	{
		DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);
	}
	if (repeats)	{
		DataHold.inthdr.hdroptions1	|= RF_DEVICE_HEADER_OPTION1_DONOT_CONCATE;
	}

	lreq.flags			= flags;
	lreq.linkcondition	= linkcondition;
	lreq.deviceoptions  = deviceoptions;
	lreq.randomno		= randomno;
	RFIntHostApp.pF_GetPowerLevel (&power, SLAVE_DEVICE_DEFAULT_POWER);
	lreq.power			= (UINT8)power;
	RFIntHostApp.pF_GetRFID (&lreq.rfid);
	RFIntHostApp.pF_GetTSID (&lreq.tsid);
	HPRF_HTONL(lreq.tsid);
	lreq.panid          = RFReceive_GetLocalDevicePanID();
	HPRF_HTONS(lreq.panid);
	memset(lreq.idstring, 0, sizeof(lreq.idstring));

	len  = sizeof(TYPE_RF_LINK_REQUEST_PACKET_DATA)-sizeof(lreq.idstring);
	memcpy ((UINT8 *) &DataHold.data, (INT8*) &lreq, len);
	slen = strlen((INT8*) pIDStr);
	if (slen >= sizeof(lreq.idstring))	{	slen = sizeof(lreq.idstring)-1;	}
	memcpy ((UINT8*) &DataHold.data[len], pIDStr, slen);
	len += slen;

	HPRF_DEBUG(DmConsolePrintf("========== %s ========== index = %i len=%hu\n", __FUNCTION__, index, sizeof(DataHold.inthdr)+len);)
	res = 	RFTransmit_PostMessageToPipeByIndex (index, (UINT8*) &DataHold, sizeof(DataHold.inthdr)+len, RF_PACKET_PRIORITY_HIGH);
	while (repeats-- && (res == SUCCESS))	{
		if (broadcast)	{	DataHold.inthdr.hdroptions1	|= RF_DEVICE_HEADER_OPTION1_BROADCAST_RPT;	}
		res = RFTransmit_PostMessageToPipeByIndex (index, (UINT8*) &DataHold, sizeof(DataHold.inthdr)+len, RF_PACKET_PRIORITY_HIGH); 
	}

	return (res);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function builds and posts the link reply packet
 *
 * \detail
 * \date        10/12/2006
 *
 * \param	    UINT8 index, UINT8 ResultCode, UINT16 RandomNo, INT8* GatewayName
 *
 * \return		UINT16 result
 *
 * \retval		returns SUCCESS/FAILURE
 */
UINT16 RFTransmit_BuildLinkReplyPacketAndPostToPipeByIndex (UINT8 index, UINT8 ResultCode, UINT16 RandomNo, INT8* pGatewayName)	{
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;
	TYPE_RF_LINK_REPLY_PACKET_DATA	*plreply;	
	UINT8	slen;
	UINT16  power;
	UINT16	len;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST;
	DataHold.inthdr.subtype 	= RF_PACKET_LINK_REPLY;
	DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);

	plreply = (TYPE_RF_LINK_REPLY_PACKET_DATA*) &DataHold.data;
	memset ((UINT8*)plreply, 0, sizeof(TYPE_RF_LINK_REPLY_PACKET_DATA));

	plreply->linkresultcode	= ResultCode;
	plreply->randomno		= RandomNo;
	plreply->addrmode		= RFReceive_GetLocalDeviceAddrMode();
	RFDesc_GetLinkQualityOfLocalDevice (index, &plreply->linkquality);
	RFIntHostApp.pF_GetPowerLevel(&power, GATEWAY_DEFAULT_POWER);
	plreply->power		    = (UINT8)power;
	RFIntHostApp.pF_GetRFID(&plreply->cnetid);	
	RFIntHostApp.pF_GetTSID(&plreply->tsid);
	HPRF_HTONL(plreply->tsid);

	if (strlen(pGatewayName) >= sizeof(plreply->gatewayname))	{	
		slen = sizeof(plreply->gatewayname)-1; 
	}	else	{	
		slen = strlen(pGatewayName);	
	}

	memcpy((INT8*)plreply->gatewayname, pGatewayName, slen);

	RFReceive_GetLocalDeviceUID ((UINT8*) (&plreply->uid));
	plreply->panid = RFReceive_GetLocalDevicePanID();
	HPRF_HTONS(plreply->panid);
	
	len = sizeof(DataHold.inthdr) + sizeof(TYPE_RF_LINK_REPLY_PACKET_DATA) - sizeof(plreply->gatewayname) + slen;

	return (RFTransmit_PostMessageToPipeByIndex (index, (UINT8*) &DataHold, len, RF_PACKET_PRIORITY_HIGH));	
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function builds and posts the device field information request packet
 *				(eg. VTZ info, ipaddress, hostname)
 *
 * \detail
 * \date        04/25/2007
 *
 * \param	    UINT8 index, UINT8 Field
 *
 * \return		UINT16 result
 *
 * \retval		returns SUCCESS/FAILURE
 */
UINT16 RFTransmit_BuildDeviceFieldDataRequestPacketAndPostToPipeByIndex (UINT8 index, UINT8 Field)	{
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF  DataHold;
	TYPE_RF_DEVICE_FIELD_PACKET_DATA_REQUEST *devfield;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST;
	DataHold.inthdr.subtype 	= RF_PACKET_DEVICE_DATA_FIELD_REQUEST;
	DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);

	devfield = (TYPE_RF_DEVICE_FIELD_PACKET_DATA_REQUEST*)&DataHold.data;
	devfield->field = Field;

	return (RFTransmit_PostMessageToPipeByIndex (index, (UINT8*) &DataHold, sizeof(DataHold.inthdr)+sizeof(TYPE_RF_DEVICE_FIELD_PACKET_DATA_REQUEST), RF_PACKET_PRIORITY_HIGH));
}


/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function builds and posts the device field (eg. VTZ info, ipaddress, hostname)
 *				packet
 *
 * \detail
 * \date        04/25/2007
 *
 * \param	    UINT8 index, UINT8 Field, UINT16 crc, UINT8 *pBuf, UINT8 BufLen
 *
 * \return		UINT16 result
 *
 * \retval		returns SUCCESS/FAILURE
 */
UINT16 RFTransmit_BuildDeviceFieldDataReplyPacketAndPostToPipeByIndex (UINT8 index, UINT8 Field, UINT16 crc, UINT8 *pBuf, UINT8 BufLen)	{
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF  DataHold;
	TYPE_RF_DEVICE_FIELD_PACKET_DATA_REPLY *devfield;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST;
	DataHold.inthdr.subtype 	= RF_PACKET_DEVICE_DATA_FIELD_REPLY;
	DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);

	devfield = (TYPE_RF_DEVICE_FIELD_PACKET_DATA_REPLY*)&DataHold.data;
	devfield->hdr.field = Field;
	devfield->hdr.crc	= crc;
	//HPRF_HTONS(devfield->hdr.crc);
	devfield->hdr.unused= 0;

	memset (devfield->data, 0, sizeof(devfield->data));
	if (BufLen > (RF_DEVICE_DATA_FIELD_MAX_SIZE)) {
		BufLen = RF_DEVICE_DATA_FIELD_MAX_SIZE;
	}
	memcpy (devfield->data, pBuf, BufLen);

	return (RFTransmit_PostMessageToPipeByIndex (index, (UINT8*) &DataHold, (sizeof(DataHold.inthdr)+(sizeof(TYPE_RF_DEVICE_FIELD_PACKET_DATA_REPLY)-(sizeof(devfield->data)-BufLen))), RF_PACKET_PRIORITY_HIGH));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief
 *
 * \detail
 * \date        10/12/2006
 *
 * \param	    UINT8 index
 *
 * \return		UINT16
 *
 * \retval		Returns SUCCESS if successful
 */
static UINT16 RFTransmit_BuildCSStatusPacketAndPostToPipeByIndex (UINT8 index)	{
	TYPE_RF_CS_STATUS_PACKET_DATA css;
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST;
	DataHold.inthdr.subtype 	= RF_PACKET_CTLSYS_CONNECT_STATUS;
	DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);

	//	clear data
	memset ((UINT8 *) &css, 0, sizeof(css));

	memcpy ((UINT8 *) &DataHold.data, (UINT8 *) &css, sizeof(css));
	return (RFTransmit_PostMessageToPipeByIndex (index, (UINT8 *) &DataHold, (sizeof(DataHold.inthdr)+sizeof(css)), RF_PACKET_PRIORITY_HIGH));
}


/**
 *
 * \author      Hazrat Shah
 *
 * \brief
 *
 * \detail
 * \date        10/12/2006
 *
 * \param	    UINT8 rfid
 *
 * \return		UINT16
 *
 * \retval		Returns SUCCESS if successful
 */
UINT16 RFTransmit_BuildCSStatusPacketAndPostToPipeByRFID (UINT8 rfid)	{
	TYPE_RF_CS_STATUS_PACKET_DATA css;
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST;
	DataHold.inthdr.subtype 	= RF_PACKET_CTLSYS_CONNECT_STATUS;
	DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);

	//	clear data
	memset ((UINT8 *) &css, 0, sizeof(css));

	memcpy ((UINT8 *) &DataHold.data, (UINT8 *) &css, sizeof(css));
	return (RFTransmit_PostMessageToPipeByRFID (rfid, (UINT8 *) &DataHold, (sizeof(DataHold.inthdr)+sizeof(css)), RF_PACKET_PRIORITY_HIGH));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function builds and posts the sleep mode packet
 *
 * \detail
 * \date        11/28/2006
 *
 * \param	    UINT8 index
 *
 * \return		UINT16
 *
 * \retval		Returns SUCCESS if successful
 */
UINT16 RFTransmit_BuildSleepModePacketAndPostToPipeByIndex (UINT8 index)	{
	TYPE_RF_SLEEP_PACKET_DATA sleep;
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST;
	DataHold.inthdr.subtype 	= RF_PACKET_ENTER_SLEEP_STATE;
	DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);

	//	clear data
	memset ((UINT8 *) &sleep.data, 0, sizeof(sleep.data));
	memcpy ((UINT8 *) &DataHold.data, (UINT8 *) &sleep.data, sizeof(sleep.data));
	return (RFTransmit_PostMessageToPipeByIndex (index, (UINT8 *) &DataHold, (sizeof(DataHold.inthdr)+sizeof(sleep.data)), RF_PACKET_PRIORITY_HIGH));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function builds and posts the power down state packet
 *
 * \detail
 * \date        11/28/2006
 *
 * \param	    UINT8 index
 *
 * \return		UINT16
 *
 * \retval		Returns SUCCESS if successful
 */
UINT16 RFTransmit_BuildPowerDownModePacketAndPostToPipeByIndex (UINT8 index)	{
	TYPE_RF_POWERDOWN_PACKET_DATA powerd;
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST;
	DataHold.inthdr.subtype 	= RF_PACKET_ENTER_POWERDOWN_STATE;
	DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);

	//	clear data
	memset ((UINT8 *) &powerd.data, 0, sizeof(powerd.data));
	memcpy ((UINT8 *) &DataHold.data, (UINT8 *) &powerd.data, sizeof(powerd.data));
	return (RFTransmit_PostMessageToPipeByIndex (index, (UINT8 *) &DataHold, (sizeof(DataHold.inthdr)+sizeof(powerd.data)), RF_PACKET_PRIORITY_HIGH));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function builds the buffer overflow packet
 *
 * \detail		The buffer overflow header and data is added to the passed on data hold buffer the data hold buffer lenght is returned
 *
 * \date        10/12/2006
 *
 * \param	    UINT8 index, UINT8 TxpipeOverFlow, UINT8 RxpipeOverFlow
 *
 * \return		UINT16 result
 *
 * \retval		Returns SUCCESS if succefully posted
 */
UINT16 RFTransmit_BuildBufferOverflowPacketAndPostToPipeByIndex (UINT8 index, UINT8 TxpipeOverFlow, UINT8 RxpipeOverFlow)	{
	TYPE_RF_BUFFER_OVERFLOW_PACKET_DATA bovf;
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF		DataHold;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST;
	DataHold.inthdr.subtype 	= RF_PACKET_BUFFER_OVERFLOW_REPORT;
	DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);

	//	Copy data
	bovf.transmitpipeoverflow 		= TxpipeOverFlow;
	bovf.receivepipeoverflow 		= RxpipeOverFlow;
	memset ((INT8 *) &bovf.unused, 0, sizeof(bovf.unused));
	memcpy ((INT8 *) &DataHold.data, (INT8 *) &bovf, sizeof(bovf));
	return (RFTransmit_PostMessageToPipeByIndex (index, (UINT8 *) &DataHold, (sizeof(DataHold.inthdr)+sizeof(bovf)), RF_PACKET_PRIORITY_HIGH));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function builds the buffer overflow packet
 *
 * \detail		The buffer overflow header and data is added to the passed on data hold buffer the data hold buffer lenght is returned
 *
 * \date        10/12/2006
 *
 * \param	    UINT8 rfid, UINT8 TxpipeOverFlow, UINT8 RxpipeOverFlow
 *
 * \return		UINT16 result
 *
 * \retval		Returns SUCCESS if succefully posted
 */
UINT16 RFTransmit_BuildBufferOverflowPacketAndPostToPipeByRFID (UINT8 rfid, UINT8 TxpipeOverFlow, UINT8 RxpipeOverFlow)	{
	TYPE_RF_BUFFER_OVERFLOW_PACKET_DATA bovf;
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF		DataHold;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST;
	DataHold.inthdr.subtype 	= RF_PACKET_BUFFER_OVERFLOW_REPORT;
	DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);

	//	Copy data
	bovf.transmitpipeoverflow 		= TxpipeOverFlow;
	bovf.receivepipeoverflow 		= RxpipeOverFlow;
	memset ((INT8 *) &bovf.unused, 0, sizeof(bovf.unused));
	memcpy ((INT8 *) &DataHold.data, (INT8 *) &bovf, sizeof(bovf));
	return (RFTransmit_PostMessageToPipeByRFID (rfid, (UINT8 *) &DataHold, (sizeof(DataHold.inthdr)+sizeof(bovf)), RF_PACKET_PRIORITY_HIGH));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function builds the communication error packet
 *
 * \detail
 * \date        09/06/2006
 *
 * \param	    UINT8 index, UINT8 Flags, UINT8 ExpectedSequenceNo, UINT8 ReceivedSequenceNo, UINT8 TotalSegments, UINT8 ExpectedSegment, UINT8 ReceivedSegment)
 *
 * \return		UINT16 result
 *
 * \retval		returns SUCCESS if successfully posted
 */
UINT16 RFTransmit_BuildCommunicationErrorPacketAndPostToPipeByIndex (UINT8 index, UINT16 Flags, UINT8 ExpectedSequenceNo, UINT8 ReceivedSequenceNo,
															  UINT8 TotalSegments, UINT8 ExpectedSegment, UINT8 ReceivedSegment)	{
	TYPE_RF_COMMUNICATION_ERROR_PACKET_DATA commerr;
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF			DataHold;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST;
	DataHold.inthdr.subtype 	= RF_PACKET_COMMUNICATION_ERROR;
	DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);

	//	Copy data
	commerr.flags				= Flags;
	//HPRF_HTONS(commerr.flags);
	commerr.expectedsequenceno 	= ExpectedSequenceNo;
	commerr.receivedsequenceno 	= ReceivedSequenceNo;
	commerr.totalsegments 		= TotalSegments;
	commerr.receivedsegment 	= ReceivedSegment;
	commerr.expectedsegment   	= ExpectedSegment;
	memset (commerr.unused, 0, sizeof(commerr.unused));

	memcpy ((UINT8 *) &DataHold.data, (UINT8 *) &commerr, sizeof(commerr));
	return (RFTransmit_PostMessageToPipeByIndex (index, (UINT8 *) &DataHold, (sizeof(DataHold.inthdr)+sizeof(commerr)), RF_PACKET_PRIORITY_HIGH));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function builds and posts the heart beat packet to the device of in descriptive entry
 *
 * \detail
 * \date        09/06/2006
 *
 * \param	    UINT8 index
 *
 * \return		UINT16 result
 *
 * \retval		returns SUCCESS if successfully posted
 */
UINT16 RFTransmit_BuildHeartBeatPacketAndPostToPipeByIndex (UINT8 index)	{
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF  DataHold;
	TYPE_RF_HEART_BEAT_PACKET_DATA    hb;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST;
	DataHold.inthdr.subtype 	= RF_PACKET_HEART_BEAT;
	DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);

	hb.linkquality = g_pRFDeviceDescriptiveTable[index].LQ.linkqualityAtLocalDevice;
	memset ((UINT8*) &hb.unused, 0, sizeof(hb.unused));
	memcpy ((UINT8 *) &DataHold.data, (UINT8*) &hb, sizeof(hb));
	return (RFTransmit_PostMessageToPipeByIndex (index, (UINT8*) &DataHold, sizeof(DataHold.inthdr)+sizeof(hb), RF_PACKET_PRIORITY_HIGH));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function builds and posts the rf net discovery request packet
 *
 * \detail
 * \date        09/06/2006
 *
 * \param	    UINT8 index, UINT32 flags, BOOL broadcast, UINT8 repeats
 *
 * \return		UINT16 result
 *
 * \retval		returns SUCCESS if successfully posted
 */
UINT16 RFTransmit_BuildRFNetDiscoverRequestPacketAndPostToPipeByIndex (UINT8 index, UINT32 flags, BOOL broadcast, UINT8 repeats)	{
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;
	TYPE_RF_NETWORK_SERIVCES_PACKET_DATA *pad = (TYPE_RF_NETWORK_SERIVCES_PACKET_DATA*)&DataHold.data;
	UINT32	len;
	UINT16  res;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST;
	DataHold.inthdr.subtype 	= RF_PACKET_NETWORK_SERVICES;
	if (broadcast)	{
		DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH|RF_DEVICE_HEADER_OPTION1_BROADCAST);
	}	else	{
		DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);
	}

	pad->msgtype = RF_PACKET_DISCOVER_NETDEV_REQUEST;
	pad->unused  = 0;
	pad->pkt.adreq.flags = flags;
	memset((UINT8*)pad->pkt.adreq.unused, 0, sizeof(pad->pkt.adreq.flags));	

	len = sizeof(DataHold.inthdr) + sizeof(TYPE_RF_NETWORK_SERIVCES_PACKET_DATA) - sizeof(pad->pkt) + sizeof (pad->pkt.adreq);

	if (repeats)	{
		DataHold.inthdr.hdroptions1	|= RF_DEVICE_HEADER_OPTION1_DONOT_CONCATE;
	}

	res = RFTransmit_PostMessageToPipeByIndex (index, (UINT8*) &DataHold, len, RF_PACKET_PRIORITY_HIGH);
	while (repeats-- && (res == SUCCESS))	{
		if (broadcast)	{	DataHold.inthdr.hdroptions1	|= RF_DEVICE_HEADER_OPTION1_BROADCAST_RPT;	}
		res = RFTransmit_PostMessageToPipeByIndex (index, (UINT8*) &DataHold, len, RF_PACKET_PRIORITY_HIGH);
	}

	return (res);	
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function builds and posts the rf net discovery reply packet
 *
 * \detail
 * \date        09/06/2006
 *
 * \param	    UINT8 index, UINT32 flags
 *
 * \return		UINT16 result
 *
 * \retval		returns SUCCESS if successfully posted
 */
UINT16 RFTransmit_BuildRFNetDiscoverReplyPacketAndPostToPipeByIndex (UINT8 index, UINT32 flags)	{
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF  DataHold;
	TYPE_RF_NETWORK_SERIVCES_PACKET_DATA *pad = (TYPE_RF_NETWORK_SERIVCES_PACKET_DATA*)&DataHold.data;
	BOOL	valid;
	UINT8	addrmode;
	UINT32	len;
	UINT8	nlen=0;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST;
	DataHold.inthdr.subtype 	= RF_PACKET_NETWORK_SERVICES;
	DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);

	pad->msgtype = RF_PACKET_DISCOVER_NETDEV_REPLY;
	pad->unused = 0;
	pad->pkt.adreply.flags = flags;
	pad->pkt.adreply.unused = 0;
	RFIntHostApp.pF_GetRFID((UINT8*) &pad->pkt.adreply.rfid);
	RFIntHostApp.pF_GetTSID((UINT32*)&pad->pkt.adreply.tsid);
	pad->pkt.adreply.power = RFReceive_GetLocalDevicePower();
	RFDesc_GetLinkQualityOfLocalDevice (index, (UINT8*)&pad->pkt.adreply.linkquality);
	RFIntHostApp.pF_GetGatewayUID_PANID_ADDRMODE (&valid, (UINT8*)&pad->pkt.adreply.gatewayuid, (UINT16*)&pad->pkt.adreply.gatewaypanid, &addrmode, (UINT8*)GATEWAY_DEFAULT_UID, GATEWAY_DEFAULT_PANID, GATEWAY_ADDR_MODE);
	RFIntHostApp.pF_GetIDString((INT8*)pad->pkt.adreply.name, &nlen);
	if (nlen >= RF_DEVICE_MAX_CRESNET_ID_STRING_LEN)	{	nlen = RF_DEVICE_MAX_CRESNET_ID_STRING_LEN-1;	}
	pad->pkt.adreply.name[nlen]=0;

	len = sizeof(DataHold.inthdr) + sizeof(TYPE_RF_NETWORK_SERIVCES_PACKET_DATA) - sizeof(pad->pkt) + sizeof (pad->pkt.adreply) - (sizeof(pad->pkt.adreply.name) - nlen);

	//return (RFTransmit_PostMessageToPipeByIndex (index, (UINT8*) &DataHold, sizeof(DataHold.inthdr)+sizeof(TYPE_RF_NETWORK_SERIVCES_PACKET_DATA), RF_PACKET_PRIORITY_HIGH));
	return (RFTransmit_PostMessageToPipeByIndex (index, (UINT8*) &DataHold, len, RF_PACKET_PRIORITY_HIGH));
}
#ifdef	RF_NETWORK_MASTER_DEVICE
/**
 *
 * \author      John Cheng
 *
 * \brief		This function builds and posts the rf net beacon request reply packet
 *
 * \detail
 * \date        12/18/2008
 *
 * \param	    UINT8 index, UINT32 flags
 *
 * \return		UINT16 result
 *
 * \retval		returns SUCCESS if successfully posted
 */
UINT16 RFTransmit_BuildRFNetBeaconReplyPacketAndPostToPipeByIndex (UINT8 index, UINT32 flags)	
{
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF  DataHold;
	TYPE_RF_NETWORK_SERIVCES_PACKET_DATA *pad = (TYPE_RF_NETWORK_SERIVCES_PACKET_DATA*)&DataHold.data;
	BOOL	valid;
	UINT8	addrmode;
	UINT32	len;
	UINT8	nlen=0;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST;
	DataHold.inthdr.subtype 	= RF_PACKET_NETWORK_SERVICES;
	DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);

	pad->msgtype = RF_PACKET_DISCOVER_NETDEV_BEACON_REPLY;
	pad->unused = 0;
	pad->pkt.adreply.flags = flags;
	pad->pkt.adreply.unused = 0;
	RFIntHostApp.pF_GetRFID((UINT8*) &pad->pkt.adreply.rfid);
	RFIntHostApp.pF_GetTSID((UINT32*)&pad->pkt.adreply.tsid);
	pad->pkt.adreply.power = RFReceive_GetLocalDevicePower();
	RFDesc_GetLinkQualityOfLocalDevice (index, (UINT8*)&pad->pkt.adreply.linkquality);
	RFIntHostApp.pF_GetGatewayUID_PANID_ADDRMODE (&valid, (UINT8*)&pad->pkt.adreply.gatewayuid, (UINT16*)&pad->pkt.adreply.gatewaypanid, &addrmode, (UINT8*)GATEWAY_DEFAULT_UID, GATEWAY_DEFAULT_PANID, GATEWAY_ADDR_MODE);
	RFIntHostApp.pF_GetIDString((INT8*)pad->pkt.adreply.name, &nlen);
	if (nlen >= RF_DEVICE_MAX_CRESNET_ID_STRING_LEN)	{	nlen = RF_DEVICE_MAX_CRESNET_ID_STRING_LEN-1;	}
	pad->pkt.adreply.name[nlen]=0;

	len = sizeof(DataHold.inthdr) + sizeof(TYPE_RF_NETWORK_SERIVCES_PACKET_DATA) - sizeof(pad->pkt) + sizeof (pad->pkt.adreply) - (sizeof(pad->pkt.adreply.name) - nlen);

	return (RFTransmit_PostMessageToPipeByIndex (index, (UINT8*) &DataHold, len, RF_PACKET_PRIORITY_HIGH));
}
#endif
/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function builds and posts the rf net remote setup cfg function 					           
 *
 * \detail		
 * \date        08/10/2007
 *
 * \param	    UINT8 index, UINT32 flags UINT8 *pdevuid, UINT16 panid, UINT8 addrmode
 *
 * \return		UINT16 result	  				
 *
 * \retval		returns SUCCESS if successfully posted			  
 */
UINT16 RFTransmit_BuildRFNetRemoteSetupPacketAndPostToPipeByIndex (UINT8 index, UINT32 flags, UINT8 *pdevuid, UINT16 devpanid, UINT8 devaddrmode,
																							  UINT8 *pgwuid, UINT16 gwpanid, UINT8 gwaddrmode)	{	
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF  DataHold;
	TYPE_RF_NETWORK_SERIVCES_PACKET_DATA *pad = (TYPE_RF_NETWORK_SERIVCES_PACKET_DATA*)&DataHold.data;
	//BOOL	valid;
	UINT32	len;

	//	Clear internal header 
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST; 
	DataHold.inthdr.subtype 	= RF_PACKET_NETWORK_SERVICES;	
	DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_BROADCAST|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);

	pad->msgtype = RF_PACKET_REMOTE_SETUP;
	pad->unused = 0;
	pad->pkt.rmts.flags = flags;
	pad->pkt.rmts.unused = 0;
	memcpy(pad->pkt.rmts.gatewayuid, pgwuid, UID_MAX_SIZE); 
	pad->pkt.rmts.gatewaypanid = gwpanid;
	pad->pkt.rmts.gatewayaddrmode =  gwaddrmode;

	memcpy(pad->pkt.rmts.deviceuid, pdevuid, UID_MAX_SIZE);  
	pad->pkt.rmts.devicepanid = devpanid;
	pad->pkt.rmts.deviceaddrmode = devaddrmode;
	memset (pad->pkt.rmts.data, 0, sizeof(pad->pkt.rmts.data));

	len = sizeof(DataHold.inthdr) + sizeof(TYPE_RF_NETWORK_SERIVCES_PACKET_DATA) - sizeof(pad->pkt) + sizeof (pad->pkt.rmts);
	
	return (RFTransmit_PostMessageToPipeByIndex (index, (UINT8*) &DataHold, len, RF_PACKET_PRIORITY_HIGH));	
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function builds and posts the rf net beacon packet 					           
 *
 * \detail		
 * \date        08/10/2007
 *
 * \param	    UINT8 index, UINT32 flags, BOOL broadcast, BOOL DonotConcate, TYPE_RFSTACK_CB *pcb
 *
 * \return		UINT16 result	  				
 *
 * \retval		returns SUCCESS if successfully posted			  
 */
 UINT16 RFTransmit_BuildRFNetDiscoverBeaconPacketAndPostToPipeByIndex (UINT8 index, UINT32 flags, BOOL broadcast, BOOL DonotConcate, TYPE_RFSTACK_CB *pcb)	{
																					 
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF  DataHold;
	TYPE_RF_NETWORK_SERIVCES_PACKET_DATA *pad = (TYPE_RF_NETWORK_SERIVCES_PACKET_DATA*)&DataHold.data;
	//BOOL	valid;
	UINT32	len;

	//	Clear internal header 
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST; 
	DataHold.inthdr.subtype 	= RF_PACKET_NETWORK_SERVICES;	
	if (broadcast)	{
		DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH|RF_DEVICE_HEADER_OPTION1_BROADCAST);
	}	else	{
		DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);
	}

	if (DonotConcate)	{
		DataHold.inthdr.hdroptions1 |= RF_DEVICE_HEADER_OPTION1_DONOT_CONCATE;
	}

	DataHold.inthdr.cb.pf		= pcb->pf;
	DataHold.inthdr.cb.par		= pcb->par;

	pad->msgtype = RF_PACKET_DISCOVER_NETDEV_BEACON;
	pad->unused = 0;
	
	// pad->pkt.beacon.flags = flags;

	len = sizeof(DataHold.inthdr) + sizeof(TYPE_RF_NETWORK_SERIVCES_PACKET_DATA) - sizeof(pad->pkt) + sizeof(pad->pkt.beacon);
	
	return (RFTransmit_PostMessageToPipeByIndex (index, (UINT8*) &DataHold, len, RF_PACKET_PRIORITY_HIGH));	
}
#ifndef	RF_NETWORK_MASTER_DEVICE
/**
 *
 * \author      John Cheng	
 *
 * \brief		This function builds and posts the rf net beacon packet 					           
 *
 * \detail		
 * \date        02/10/2009
 *
 * \param	    UINT8 index, UINT32 flags, BOOL broadcast
 *
 * \return		UINT16 result	  				
 *
 * \retval		returns SUCCESS if successfully posted			  
 */
UINT16 RFTransmit_BuildRFBeaconReqPacketAndPostToPipeByIndex (UINT8 index, UINT32 flags, BOOL broadcast)	
{
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF  DataHold;
	TYPE_RF_NETWORK_SERIVCES_PACKET_DATA *pad = (TYPE_RF_NETWORK_SERIVCES_PACKET_DATA*)&DataHold.data;
	//BOOL	valid;
	UINT32	len;
	//	Clear internal header 
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build rf device internal packet header
	DataHold.inthdr.type    	= RF_PACKET_TYPE_MCP_REQUEST; 
	DataHold.inthdr.subtype 	= RF_PACKET_NETWORK_SERVICES;	
	if (broadcast)	{
		DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH|RF_DEVICE_HEADER_OPTION1_BROADCAST);
	}	else	{
		DataHold.inthdr.hdroptions1	= (RF_DEVICE_HEADER_OPTION1_REQ_ACK|RF_DEVICE_HEADER_OPTION1_PRIORITY_HIGH);
	}

	//DataHold.inthdr.cb.pf		= pcb->pf;
	//DataHold.inthdr.cb.par		= pcb->par;

	pad->msgtype = RF_PACKET_DISCOVER_NETDEV_BEACON_REQ;
	pad->unused = 0;
	
	pad->pkt.beacon.flags = flags;

	len = sizeof(DataHold.inthdr) + sizeof(TYPE_RF_NETWORK_SERIVCES_PACKET_DATA) - sizeof(pad->pkt) + sizeof(pad->pkt.beacon);
	
	return (RFTransmit_PostMessageToPipeByIndex (index, (UINT8*) &DataHold, len, RF_PACKET_PRIORITY_HIGH));	
}
#endif
/////////////////////////////////////////////////////////////////////////////////////////
//	Building ASP packets
/////////////////////////////////////////////////////////////////////////////////////////
/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions builds the power level frame and posts to the rf transmit pipe
 *
 * \detail
 *
 * \date        09/13/2006
 *
 * \param	    UINT16 power, BOOL TranslatePower
 *
 * \return		UINT16
 *
 * \retval		returns SUCCESS if successful
 */
UINT16	RFTransmit_BuildPowerLevelFrameAndPostToPipe (UINT16 power, BOOL TranslatePower)	{
	appToAspMsg_t *pMsg;
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build packet header
	DataHold.inthdr.type 		= RF_PACKET_TYPE_ASP_REQUEST;
	DataHold.inthdr.subtype 	= gAppAspSetPowerLevelReq_c;

	pMsg = (appToAspMsg_t *) &DataHold.data;

    /*	This is a ASP-power.req command */
    pMsg->msgType = gAppAspSetPowerLevelReq_c;

	if (TranslatePower) {
		//	translate power level to PHY limits from analog value
		pMsg->msgData.aspSetPowerLevelReq.powerLevel = RFReceive_TranslatePowerToPHYLimits(power);
	}
	else {
		pMsg->msgData.aspSetPowerLevelReq.powerLevel = (UINT8)power;
	}

	return (RFTransmit_PostMessageToPipeByIndex (RFDesc_GetASPDescriptiveTableIndex(), (UINT8 *) &DataHold , sizeof(DataHold.inthdr)+sizeof(appToAspMsg_t), RF_PACKET_PRIORITY_HIGH));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions sets channel where the TELC tests can be conducted
 *
 * \detail
 *
 * \date        12/18/2006
 *
 * \param	    UINT8 channel
 *
 * \return		UINT16
 *
 * \retval		returns SUCCESS if successfully posted
 */
UINT16	RFTransmit_BuildTELCTestChannelMessageAndPostToPipe (UINT8 channel)	{
	appToAspMsg_t *pMsg;
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build packet header
	DataHold.inthdr.type 		= RF_PACKET_TYPE_ASP_REQUEST;
	DataHold.inthdr.subtype 	= gAppAspTelecSetChannelReq_c;
	DataHold.inthdr.req			= gAppAspTelecSetChannelReq_c;

	pMsg = (appToAspMsg_t *) &DataHold.data;

    /*	This is a ASP-TELC test channel request command */
    pMsg->msgType = gAppAspTelecSetChannelReq_c;
    pMsg->msgData.aspTelecSetChannelReq.logicalChannel = channel;

	return (RFTransmit_PostMessageToPipeByIndex (RFDesc_GetASPDescriptiveTableIndex(), (UINT8 *) &DataHold , sizeof(DataHold.inthdr)+sizeof(appToAspMsg_t), RF_PACKET_PRIORITY_HIGH));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions starts the TELC tests
 *
 * \detail		This function performs the TELC tests, the stack must be reset after the tests are complete
 *
 * \date        12/18/2006
 *
 * \param	    UINT8 mode
 *
 * \return		UINT16
 *
 * \retval		returns SUCCESS if successfully posted
 */
UINT16	RFTransmit_BuildTELCTestMessageAndPostToPipe (UINT8 mode)	{
	appToAspMsg_t *pMsg;
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build packet header
	DataHold.inthdr.type 		= RF_PACKET_TYPE_ASP_REQUEST;
	DataHold.inthdr.subtype 	= gAppAspTelecTestReq_c;
	DataHold.inthdr.req		    = gAppAspTelecTestReq_c;

	pMsg = (appToAspMsg_t *) &DataHold.data;

    /*	This is a ASP-TELC test request command */
    pMsg->msgType = gAppAspTelecTestReq_c;
    pMsg->msgData.aspTelecTestReq.mode = mode;

	return (RFTransmit_PostMessageToPipeByIndex (RFDesc_GetASPDescriptiveTableIndex(), (UINT8 *) &DataHold , sizeof(DataHold.inthdr)+sizeof(appToAspMsg_t), RF_PACKET_PRIORITY_HIGH));
}

/////////////////////////////////////////////////////////////////////////////////////////
//	Building MLME packets
/////////////////////////////////////////////////////////////////////////////////////////
/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions will post a message to start the scan process of the
 * 				specified type in the MAC.
 *
 * \detail
 *
 * \date        09/13/2006
 *
 * \param	    UINT8 scanType, UINT8 scanDuration, UINT32 ChannelMask
 *
 * \return		UINT16
 *
 * \retval		returns SUCCESS if successful
 */
UINT16	RFTransmit_BuildAppScanPacketAndPostToPipe (UINT8 scanType, UINT8 scanDuration, UINT32 ChannelMask)	{
	mlmeMessage_t *pMsg;
  	mlmeScanReq_t *pScanReq;
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	setup the call back function
	//DataHold.pcbf = pcallback;

	//	Build packet header
	DataHold.inthdr.type		= RF_PACKET_TYPE_MLM_REQUEST;
	DataHold.inthdr.subtype		= gMlmeScanReq_c;
	DataHold.inthdr.req			= scanType;

	pMsg = (mlmeMessage_t*) &DataHold.data;

    /* This is a MLME-START.req command */
    pMsg->msgType = gMlmeScanReq_c;
    /* Create the Start request message data. */
    pScanReq = &pMsg->msgData.scanReq;
    /* gScanModeED_c, gScanModeActive_c, gScanModePassive_c, or gScanModeOrphan_c */
    pScanReq->scanType = scanType;

    /* ChannelsToScan & 0xFF - LSB, always 0x00 */
    pScanReq->scanChannels[0] = 0x00;	//	(uint8_t)((ChannelMask)     & 0xFF);
    /* ChannelsToScan>>8 & 0xFF  */
    pScanReq->scanChannels[1] = (uint8_t)((ChannelMask>>8)  & 0xFF);
    /* ChannelsToScan>>16 & 0xFF  */
    pScanReq->scanChannels[2] = (uint8_t)((ChannelMask>>16) & 0xFF);
    /* ChannelsToScan>>24 & 0xFF - MSB */
    pScanReq->scanChannels[3] = (uint8_t)((ChannelMask>>24) & 0xFF);

    /* Duration per channel 0-14 (dc). T[sec] = (16*960*((2^dc)+1))/1000000.
       A scan duration of 3 on 16 channels approximately takes 2 secs. */
    pScanReq->scanDuration = scanDuration;

	return (RFTransmit_PostMessageToPipeByIndex (RFDesc_GetMLMEDescriptiveTableIndex(), (UINT8 *) &DataHold, sizeof(DataHold.inthdr)+sizeof(mlmeMessage_t), RF_PACKET_PRIORITY_HIGH));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions builds the MLM associate request message and posts it to the pipe
 *
 * \detail
 *
 * \date        10/13/2006
 *
 * \param	    UINT8 *puid, UINT16 panid
 *
 * \return		UINT16
 *
 * \retval		returns SUCCESS if successfully posted
 */
UINT16	RFTransmit_BuildAssociateRequestMessageAndPostToPipe (UINT8 *puid, UINT16 panid)	{
	mlmeMessage_t *pMsg;
	mlmeAssociateReq_t *pAssocReq;
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;
	UINT16 pid = ConvUINTToLittleIndian (panid);

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build packet header
	DataHold.inthdr.type 		= RF_PACKET_TYPE_MLM_REQUEST;
	DataHold.inthdr.subtype 	= gMlmeAssociateReq_c;

	pMsg = (mlmeMessage_t *) &DataHold.data;

    /*	This is a MLME-ASSOCIATE.req command. */
    pMsg->msgType = gMlmeAssociateReq_c;

	/*	Create the Associate request message data. */
    pAssocReq = &pMsg->msgData.associateReq;

    /*	Use the coordinator info we got from the Active Scan. */
    memcpy(&pAssocReq->coordAddress, puid, UID_MAX_SIZE);
    memcpy(&pAssocReq->coordPanId,   &pid, 2);
    pAssocReq->coordAddrMode	= RFReceive_GetLocalDeviceAddrMode();
    pAssocReq->logicalChannel   = RFReceive_GetLocalDeviceChannel();
    pAssocReq->securityEnable   = RFReceive_GetLocalDeviceSecurity();
    /*	We want the coordinator to assign a short address to us. */
    pAssocReq->capabilityInfo     = gCapInfoAllocAddr_c;

    //	Post MLM message to the pipe
	return (RFTransmit_PostMessageToPipeByIndex (RFDesc_GetMLMEDescriptiveTableIndex(), (UINT8 *) &DataHold, sizeof(DataHold.inthdr)+sizeof(mlmeMessage_t), RF_PACKET_PRIORITY_HIGH));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions builds the MLM associate response message and posts it to the pipe
 *
 * \detail
 *
 * \date        10/13/2006
 *
 * \param	    UINT8 *puid
 *
 * \return		UINT16
 *
 * \retval		returns SUCCESS if successfully posted
 */
UINT16	RFTransmit_BuildAssociateResponseMessageAndPostToPipe (UINT8 *puid)	{
  	mlmeMessage_t      *pMsg;
  	mlmeAssociateRes_t *pAssocRes;
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build packet header
	DataHold.inthdr.type 		= RF_PACKET_TYPE_MLM_REQUEST;
	DataHold.inthdr.subtype 	= gMlmeAssociateRes_c;

	pMsg = (mlmeMessage_t *) &DataHold.data;

    /*	This is a MLME-ASSOCIATE.req command. */
    pMsg->msgType = gMlmeAssociateRes_c;

    /* Create the Associate response message data. */
    pAssocRes = &pMsg->msgData.associateRes;

    /* Assign a short address to the device. In this example we simply
       choose 0x0001. Though, all devices and coordinators in a PAN must have
       different short addresses. However, if a device do not want to use
       short addresses at all in the PAN, a short address of 0xFFFE must
       be assigned to it. */
    /* A short address of 0xfffe means that the device is granted access to
      the PAN (Associate successful) but that long addressing is used.*/
    pAssocRes->assocShortAddress[0] = 0xFE;
    pAssocRes->assocShortAddress[1] = 0xFF;

    /* Get the 64 bit address of the device requesting association. */
    FLib_MemCpy(&pAssocRes->deviceAddress, puid, UID_MAX_SIZE);
    /* Association granted. May also be gPanAtCapacity_c or gPanAccessDenied_c. */
    pAssocRes->status = gSuccess_c;
    /* Do not use security */
    pAssocRes->securityEnable = FALSE;

    //	Post MLM message to the pipe
    return (RFTransmit_PostMessageToPipeByIndex (RFDesc_GetMLMEDescriptiveTableIndex(), (UINT8 *) &DataHold, sizeof(DataHold.inthdr)+sizeof(mlmeMessage_t), RF_PACKET_PRIORITY_HIGH));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions builds the MLM disassociate request message and posts it to the pipe
 *
 * \detail
 *
 * \date        12/13/2006
 *
 * \param	    UINT8 *puid, UINT8 reason
 *
 * \return		UINT16
 *
 * \retval		returns SUCCESS if successfully posted
 */
UINT16	RFTransmit_BuildDisAssociateRequestMessageAndPostToPipe (UINT8 *puid, UINT8 reason)	{
	mlmeMessage_t *pMsg;
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;

	//	Clear header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build packet header
	DataHold.inthdr.type 		= RF_PACKET_TYPE_MLM_REQUEST;
	DataHold.inthdr.subtype 	= gMlmeDisassociateReq_c;

	pMsg = (mlmeMessage_t *) &DataHold.data;

    /*	This is a MLME-DISASSOCIATE.req command. */
    pMsg->msgType = gMlmeDisassociateReq_c;

    memcpy(&pMsg->msgData.disassociateReq.deviceAddress, puid, UID_MAX_SIZE);
    pMsg->msgData.disassociateReq.securityEnable     = RFReceive_GetLocalDeviceSecurity();
	pMsg->msgData.disassociateReq.disassociateReason = reason;

	//	Post MLM message to the pipe
	return (RFTransmit_PostMessageToPipeByIndex (RFDesc_GetMLMEDescriptiveTableIndex(), (UINT8 *) &DataHold, sizeof(DataHold.inthdr)+sizeof(mlmeMessage_t), RF_PACKET_PRIORITY_HIGH));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions updates the given pib attribute value in stack
 *
 * \detail
 *
 * \date        12/13/2006
 *
 * \param	    UINT8 Attribute, UINT8 *pValue, UINT8 Len
 *
 * \return		UINT16
 *
 * \retval		returns SUCCESS if successfully posted
 */
UINT16	RFTransmit_BuildSetPIBAttributeMessageAndPostToPipe (UINT8 Attribute, UINT8 *pValue, UINT8 Len)	{
	mlmeMessage_t *pMsg;
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build packet header
	DataHold.inthdr.type 		= RF_PACKET_TYPE_MLM_REQUEST;
	DataHold.inthdr.subtype 	= gMlmeSetReq_c;
	DataHold.inthdr.req			= Attribute;

	memcpy ((UINT8*) &PIBASet[Attribute], pValue, Len);

	pMsg = (mlmeMessage_t *) &DataHold.data;
	pMsg->msgType = gMlmeSetReq_c;
	pMsg->msgData.setReq.pibAttribute      = Attribute;
	pMsg->msgData.setReq.pibAttributeValue = (UINT8*) &PIBASet[Attribute];

	//	Post MLM message to the pipe
	return (RFTransmit_PostMessageToPipeByIndex (RFDesc_GetMLMEDescriptiveTableIndex(), (UINT8 *) &DataHold, sizeof(DataHold.inthdr)+sizeof(mlmeMessage_t), RF_PACKET_PRIORITY_HIGH));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions will post the channel selection message to the pipe
 *				which will be transmitted to the rf ASP layer
 *
 * \detail
 *
 * \date        09/13/2006
 *
 * \param	    UINT8 channel
 *
 * \return		UINT16 result
 *
 * \retval		returns SUCCESS if successful
 */
UINT16	RFTransmit_BuildChannelSelectionFrameAndPostToPipe (UINT8 channel)	{
	return (RFTransmit_BuildSetPIBAttributeMessageAndPostToPipe(gMPibLogicalChannel_c, &channel, 1));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions sets the rf stack to idle state after packet reception
 *
 * \detail
 *
 * \date        01/10/2007
 *
 * \param	    UINT8 Value (0=disable, 1=enable, 2=device off, coodinator on, 3=device on, coordinator off)
 *
 * \return		UINT16 result
 *
 * \retval		returns SUCCESS if successful
 */
UINT16	RFTransmit_BuildSettoIdleFrameAndPostToPipe (UINT8 value)	{
	return (RFTransmit_BuildSetPIBAttributeMessageAndPostToPipe(gMPibRxOnWhenIdle_c, &value, 1));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions sets the receiver into premiscious mode (in this mode all packet types are passed to application)
 *
 * \detail
 *
 * \date        12/13/2006
 *
 * \param	    BOOL Enable
 *
 * \return		return
 *
 * \retval		returns SUCCESS if successfully posted
 */
UINT16	RFTransmit_BuildPromiscuousMessageAndPostToPipe (BOOL Enable)	{
	UINT8 enb = (UINT8) Enable;
	return (RFTransmit_BuildSetPIBAttributeMessageAndPostToPipe(gMPibPromiscuousMode_c, &enb, 1));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions updates the UID in the rf stack
 *
 * \detail
 *
 * \date        12/13/2006
 *
 * \param	    UINT8 *pNewUID
 *
 * \return		return
 *
 * \retval		returns SUCCESS if successfully posted
 */
UINT16	RFTransmit_BuildUIDMessageAndPostToPipe (UINT8 *pNewUID)	{
	return (RFTransmit_BuildSetPIBAttributeMessageAndPostToPipe(gMPibCoordExtendedAddress_c, pNewUID, UID_MAX_SIZE));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions updates the PANID in the rf stack
 *
 * \detail
 *
 * \date        12/13/2006
 *
 * \param	    UINT16 NewPANID
 *
 * \return		UINT16
 *
 * \retval		returns SUCCESS if successfully posted
 */
UINT16	RFTransmit_BuildPANIDMessageAndPostToPipe (UINT16 NewPANID)	{
	UINT16 panid = ConvUINTToLittleIndian (NewPANID);
	return (RFTransmit_BuildSetPIBAttributeMessageAndPostToPipe(gMPibPanId_c, (UINT8*) &panid, 2));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions updates the short address in the rf stack
 *
 * \detail
 *
 * \date        12/13/2006
 *
 * \param	    UINT16 ShortAddr
 *
 * \return		UINT16
 *
 * \retval		returns SUCCESS if successfully posted
 */
UINT16	RFTransmit_BuildShortAddrMessageAndPostToPipe (UINT16 ShortAddr)	{
	UINT16 addr = ConvUINTToLittleIndian (ShortAddr);
	return (RFTransmit_BuildSetPIBAttributeMessageAndPostToPipe(gMPibCoordShortAddress_c, (UINT8*) &addr, 2));
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions sets the receiver into premiscious mode (in this mode all packet types are passed to application)
 *
 * \detail
 *
 * \date        12/13/2006
 *
 * \param	    UINT8 Attribute, TYPE_RFSTACK_CB *pcb
 *
 * \return		UINT16
 *
 * \retval		returns SUCCESS if successfully posted
 */
UINT16	RFTransmit_BuildGetPIBAttributeMessageAndPostToPipe (UINT8 Attribute, TYPE_RFSTACK_CB *pcb)	{
	mlmeMessage_t *pMsg;
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;
	UINT16	res;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	if (Attribute <= gMPibAclEntrySecuritySuite_c)	{
		//	Build packet header
		DataHold.inthdr.type 	= RF_PACKET_TYPE_MLM_REQUEST;
		DataHold.inthdr.subtype = gMlmeGetReq_c;
		DataHold.inthdr.req		= Attribute;
		DataHold.inthdr.cb.pf	= pcb->pf;
		DataHold.inthdr.cb.par	= pcb->par;

		pMsg = (mlmeMessage_t *) &DataHold.data;

		pMsg->msgType = gMlmeGetReq_c;
		pMsg->msgData.setReq.pibAttribute      = Attribute;
		pMsg->msgData.setReq.pibAttributeValue = (UINT8*) &PIBAGet[Attribute-gMPibRole_c].value;

		//	Post MLM message to the pipe
		res = RFTransmit_PostMessageToPipeByIndex (RFDesc_GetMLMEDescriptiveTableIndex(), (UINT8 *) &DataHold, sizeof(DataHold.inthdr)+sizeof(mlmeMessage_t), RF_PACKET_PRIORITY_HIGH);
	}	else	{
		res = FAILURE;
	}
	return (res);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions reads the PIB attribute values (freescale stack)
 *
 * \detail
 *
 * \date        12/18/2006
 *
 * \param	    UINT8 Attribute, UINT8 *pbuf, UINT8 *plen
 *
 * \return		UINT16
 *
 * \retval		returns SUCCESS if successfully posted
 */
UINT16	RFTransmit_ReadPIBAttributeValue(UINT8 Attribute, UINT8 *pbuf, UINT8 *plen)	{
	UINT16	res;

	if (Attribute <= gMPibAclEntrySecuritySuite_c)	{

		//	todo: add switch and copy number of params based on attribute type
		*plen = 8;
		memcpy (pbuf, (UINT8*) &PIBAGet[Attribute-gMPibRole_c].value, *plen);

		res = SUCCESS;
	}	else	{
		res = FAILURE;
	}

	return (res);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions resets the rf stack MAC layer
 *
 * \detail
 *
 * \date        12/18/2006
 *
 * \param	    BOOL SetToDefault (1=Set to default, 0=Keep existing settings)
 *
 * \return		UINT16
 *
 * \retval		returns SUCCESS if successfully posted
 */
UINT16	RFTransmit_BuildResetMACMessageAndPostToPipe (BOOL SetToDefault)	{
	mlmeMessage_t *pMsg;
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF DataHold;

	//	Clear internal header
	memset ((UINT8*) &DataHold.inthdr, 0, sizeof(DataHold.inthdr));

	//	Build packet header
	DataHold.inthdr.type 		= RF_PACKET_TYPE_MLM_REQUEST;
	DataHold.inthdr.subtype 	= gMlmeResetReq_c;
	DataHold.inthdr.req			= gMlmeResetReq_c;

	pMsg = (mlmeMessage_t *) &DataHold.data;

    /*	This is a rf stack MAC reset request */
	pMsg->msgType = gMlmeResetReq_c;
	pMsg->msgData.resetReq.setDefaultPib = SetToDefault;

	//	increase SMAC reset counter
	gMStatp.SMACResetCntr++;

    //	Post MLM message to the pipe
	return (RFTransmit_PostMessageToPipeByIndex (RFDesc_GetMLMEDescriptiveTableIndex(), (UINT8 *) &DataHold, sizeof(DataHold.inthdr)+sizeof(mlmeMessage_t), RF_PACKET_PRIORITY_HIGH));
}

/////////////////////////////////////////////////////////////////////////////////////////
//	Other Functions
/////////////////////////////////////////////////////////////////////////////////////////
/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions sets the "TriggerRFTransmitTask" event flag for the device with given index
 *
 * \detail
 *
 * \date        10/11/2006
 *
 * \param	    UINT8 index
 *
 * \return		None
 *
 * \retval		None
 */
void	RFTransmit_TriggerRFTransmitTaskEvent (UINT8 index)	{
#if defined (OS_NUCLEUS)
	NU_Set_Events(&RFTransTask.TriggerRFTransmitTaskSubEvents[index/32], (1<<(index%32)), NU_OR);
	NU_Set_Events(&RFTransTask.TriggerRFTransmitTaskMainEvents, 	     (1<<(index/32)), NU_OR);
#elif defined (OS_WINCE)
	//SetEvent(RFTransTask.hEvents[index]);
	EnterCriticalSection(&RFTransTask.RfTransEventCriticalSection);
	RFTransTask.RfTransmitEventGroup[index/32] |= (1 << (index % 32));
	LeaveCriticalSection(&RFTransTask.RfTransEventCriticalSection);
	SetEvent(RFTransTask.hRfTransmitEvent);
#endif
    size_t main_event = index/32;
    HPRF_DEBUG(DmConsolePrintf("========== %s ========== %pS index = %i\n", __FUNCTION__, __builtin_return_address(0), index);)
	OsSetEvent(RFTransTask.TriggerRFTransmitTaskSubEvents[main_event], (1<<index%32));
	OsSetEvent(RFTransTask.TriggerRFTransmitTaskMainEvents, 1 << main_event);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions sets the "RFStackBufferAvailable" event flag
 *
 * \detail
 *
 * \date        10/11/2006
 *
 * \param	    None
 *
 * \return		None
 *
 * \retval		None
 */
void	RFTransmit_RFStackBufferAvailableEvent (void)	{
#if defined (OS_NUCLEUS)
	NU_Set_Events(&RFTransTask.rfstackbufferavailableevents, RF_STACK_BUFFER_AVAILABLE_EVENT, NU_OR);
#elif defined (OS_WINCE)
	SetEvent(RFTransTask.hRfStackbufferavailableevents);
#endif
	OsSignalEvent(RFTransTask.hRfStackBufferAvailableEvents);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions extracts all messages from the pipe, it informs the message posting functions
 *
 * \detail
 *
 * \date        05/28/2007
 *
 * \param	    UINT8 index
 *
 * \return		UINT16
 *
 * \retval		SUCCESS/FAILURE
 */
UINT16	RFTransmit_ResetTxBuffers (UINT8 index)	{
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = (g_pRFDeviceDescriptiveTable+index);
	UINT16	res = FAILURE;
	TYPE_RF_DEVICE_RF_DATA_HOLD_BUF	DataHold;
#if defined (OS_WINCE)
	DWORD dwFlags;			// just for reading sake, not used.
#endif

	if (index < g_MaxDescriptiveTableEntries)	{
		//	get messages from the pipe
#if defined (OS_NUCLEUS)
		while (NU_Receive_From_Pipe(&pdst->pipe, (void*) &(DataHold.inthdr), RF_DEVICE_PIPE_MAX_MSG_LEN, (UINT32*) &DataHold.len, NU_NO_SUSPEND) == NU_SUCCESS)	{
#elif defined (OS_WINCE)
		while ((ReadMsgQueue(pdst->QueueReadHandle, (void *) &(DataHold.inthdr), sizeof (TYPE_RF_DEVICE_RFTX_INTERNAL_HDR), (UINT32*) &DataHold.len, OS_NO_SUSPEND, &dwFlags)) == TRUE)
		{
#endif
		while(1) {
		    DataHold.len = OsDequeueBytesWait(pdst->QueueHandle, (void*)&DataHold.inthdr, sizeof (TYPE_RF_DEVICE_RFTX_INTERNAL_HDR), OS_NO_WAIT);
		    HPRF_QUEUE_DEBUG(DmConsolePrintf("--- %s:%i <--- dequeued %li bytes from desc[%i]\n", \
                    __FUNCTION__, __LINE__, DataHold.len, index);)
		    if(DataHold.len <= 0) {
		        break;
		    }
			//	make call-back if programmed
			if (DataHold.inthdr.cb.pf && DataHold.len)	{
				DataHold.inthdr.cb.pf (DataHold.inthdr.cb.par, gTransactionOverflow_c);
			}
		}
#ifdef	RF_NETWORK_MASTER_DEVICE

#else
		//	reset rf MAC buffers
		//	RFReceiver_RFStackMsgQueInit();
#endif

		res = SUCCESS;
	}

	return(res);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions
 *
 * \detail
 *
 * \date        05/28/2007
 *
 * \param	    UINT8 index
 *
 * \return		UINT16
 *
 * \retval		SUCCESS/FAILURE
 */
UINT16	RFTransmit_ResetTxDataHoldBuffer (UINT8 index)	{
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = (g_pRFDeviceDescriptiveTable+index);
	UINT16	res = FAILURE;

	if (index < g_MaxDescriptiveTableEntries)	{
		//	make call back if first segment of a sgemented packet 
		//	make call back if non-segmented packet
		if (pdst->TxBuf.DataHold.len && ((pdst->TxBuf.DataHold.totalsegment <= 1) || ((pdst->TxBuf.DataHold.totalsegment > 1) && (pdst->TxBuf.DataHold.segmentno == 1))))	{
			if (pdst->TxBuf.DataHold.inthdr.cb.pf)	{
				pdst->TxBuf.DataHold.inthdr.cb.pf (pdst->TxBuf.DataHold.inthdr.cb.par, gDisableTrxFailure_c);
			}
			pdst->TxBuf.DataHold.len = 0;
		}
		res = SUCCESS;
	}

	return(res);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions checks if packets timetolive has expired and then makes the
 *				call-back to the initiator if set 
 *
 * \detail
 *
 * \date        06/05/2007
 *
 * \param	    UINT8 index
 *
 * \return		UINT16
 *
 * \retval		SUCCESS/FAILURE
 */
UINT16	RFTransmit_MCPPacketTimeToLiveExpirationChk (UINT8 index, BOOL *allpktsexpired)	{
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = (g_pRFDeviceDescriptiveTable+index);
	UINT32	len = 0;
	UINT32	cnt = 0, exppkts = 0;
	TYPE_EXTERNAL_RF_PACKET_HEADER *ppkthdr;
	UINT16	res = FAILURE;
	*allpktsexpired = FALSE;

	if (index < g_MaxSupportedRFDevices)	{
		res = SUCCESS;
		while (len < (pdst->TxBuf.Transmit.len-sizeof(TYPE_RF_PAYLOAD_HEADER)))	{
			//	point to packet header 
			ppkthdr = (TYPE_EXTERNAL_RF_PACKET_HEADER*)&pdst->TxBuf.Transmit.payld.data[len];
			
			if (SUCCESS != RFTransmit_CheckifMessagehastimetolive (pdst->TxBuf.Transmit.txcb[cnt].timetolivestart, pdst->TxBuf.Transmit.txcb[cnt].timetolive))	{
				//	On a segmented packet if the first segment has expired the rest of the packet segmenetd are discarded  
				//	and a call-back is made, the first segment is taged is invalid and the timetolive is cleared. 
				//	On a non-segmented packet, if the packet is expiration the call-back is made, the packet is tagged as
				//	invalid and the timetolive is cleared
				//	make call-back on the first expired segment
				if	(ppkthdr->segmentno == 1)	{
					if (pdst->TxBuf.Transmit.txcb[cnt].pf)	{
						//	make call-back on the first expired segment
						pdst->TxBuf.Transmit.txcb[cnt].pf(pdst->TxBuf.Transmit.txcb[cnt].par, gTransactionExpired_c);
						//	clear call-back 
						pdst->TxBuf.Transmit.txcb[cnt].pf = NULL;	
					}
					//	change the subpacket cmd type as discarded 
					pdst->TxBuf.Transmit.payld.data[pdst->TxBuf.Transmit.txcb[cnt].index+1] = RF_PACKET_DISCARD;
					if (ppkthdr->totalsegments > 1)	{
						//	clear the rest of the segmented packet from the hold buffer, it has expired 
						pdst->TxBuf.DataHold.len = 0;
					}
				}
				
				//	clear timetolive so, it does not get reprocessed
				pdst->TxBuf.Transmit.txcb[cnt].timetolive = 0;
				exppkts++;
				//	enhancement: remove expired sub-packet out of the transmit buf and cleanup pointers to this entry 
			}
			
			if (++cnt >= RF_TRANSMIT_BUF_MAX_NO_OF_CBS)	{	break;	}
			len += (ppkthdr->length+sizeof(ppkthdr->length));
		}
		if ((cnt == exppkts) && exppkts)	{	*allpktsexpired = TRUE;	}
	}		

	return (res);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions posts call backs for given descriptive table entry for MCP messages
 *				It handles call-backs for concatenated, segmented and non-segmented packets
 *
 * \detail
 *
 * \date        06/05/2007
 *
 * \param	    UINT8 index
 *
 * \return		UINT16
 *
 * \retval		SUCCESS/FAILURE
 */
UINT16	RFTransmit_MCPPacketResultCodeCallBack (UINT8 index)	{
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = (g_pRFDeviceDescriptiveTable+index);
	UINT32	len = 0;
	UINT32	cnt = 0;
	TYPE_EXTERNAL_RF_PACKET_HEADER *ppkthdr;
	UINT16	res = FAILURE;

	if (index < g_MaxSupportedRFDevices)	{
		res = SUCCESS;
		while (len < (pdst->TxBuf.Transmit.len-sizeof(TYPE_RF_PAYLOAD_HEADER)))	{
			//	point to packet header 
			ppkthdr = (TYPE_EXTERNAL_RF_PACKET_HEADER*)&pdst->TxBuf.Transmit.payld.data[len];
			
			if (pdst->TxBuf.Transmit.txcb[cnt].pf)	{				
				//	1- On segmented packet when the last segment is sent or if any segment fails 
				if (  ((pdst->resultcode != gSuccess_c) || (ppkthdr->totalsegments == ppkthdr->segmentno)))	{ 
					pdst->TxBuf.Transmit.txcb[cnt].pf(pdst->TxBuf.Transmit.txcb[cnt].par, pdst->resultcode);
				}				
				pdst->TxBuf.Transmit.txcb[cnt].pf = NULL;
			}
			
			if (++cnt >= RF_TRANSMIT_BUF_MAX_NO_OF_CBS)	{	break;	}
			len += (ppkthdr->length+sizeof(ppkthdr->length));
		}
	}

	return (res);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions posts call backs for given descriptive table entry for MLME messages
 *
 * \detail
 *
 * \date        06/06/2007
 *
 * \param	    UINT8 index
 *
 * \return		UINT16
 *
 * \retval		SUCCESS/FAILURE
 */
UINT16	RFTransmit_MLMEPacketResultCodeCallBack (UINT8 index)	{
	UINT16	res = FAILURE;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = (g_pRFDeviceDescriptiveTable+index);

	if (index == RFDesc_GetMLMEDescriptiveTableIndex())	{
		if (pdst->TxBuf.Transmit.len && pdst->TxBuf.Transmit.txcb[0].pf)	{
			pdst->TxBuf.Transmit.txcb[0].pf(pdst->TxBuf.Transmit.txcb[0].par, pdst->resultcode);
		}
		res = SUCCESS;
	}

	return (res);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions posts call backs for given descriptive table entry for ASP messages
 *
 * \detail
 *
 * \date        06/06/2007
 *
 * \param	    UINT8 index
 *
 * \return		UINT16
 *
 * \retval		SUCCESS/FAILURE
 */
UINT16	RFTransmit_ASPPacketResultCodeCallBack (UINT8 index)	{
	UINT16	res = FAILURE;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = (g_pRFDeviceDescriptiveTable+index);

	if (index == RFDesc_GetASPDescriptiveTableIndex())	{
		if (pdst->TxBuf.Transmit.len && pdst->TxBuf.Transmit.txcb[0].pf)	{
			pdst->TxBuf.Transmit.txcb[0].pf(pdst->TxBuf.Transmit.txcb[0].par, pdst->resultcode);
		}
		res = SUCCESS;
	}

	return (res);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This functions retriggers Low Priority Message Post Delay
 *
 * \detail		The porpose of this delay is to slow down the low priorty messages transmission
 *				to the touch panels whenever a message is received over the air from the touch panel.
 *				This senaior would typically occur when the gateway is sending an image server file to
 *				the touch panel and the user presses button on the touch panel. The user action is then
 *				given precedence.
 *
 *
 *
 * \date        08/03/2007
 *
 * \param	    void
 *
 * \return		void
 *
 * \retval				 	  
 */				
void	RFTransmit_ReTriggerLowPriorityMessagePostDelay (void)	{	
	if ((LowPriorityMessagePostingDelay + 2)  < 20 )	{
		LowPriorityMessagePostingDelay += 2;
	}	else	{
		LowPriorityMessagePostingDelay = 20;
	}
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function checks whether the message has valid timestamp
 *
 * \detail		 
 *
 * \date        08/03/2007
 *
 * \param	    UINT32 start, UINT32 timetolive
 *
 * \return		INT32
 *
 * \retval		SUCCESS, FAILURE		 	  
 */				
INT32	RFTransmit_CheckifMessagehastimetolive (UINT32 start, UINT32 timetolive)	{	
	INT32 res = SUCCESS;
	if (timetolive)	{
		if ((OS_RetrieveClock()-start)>timetolive)	{
			res = FAILURE;
		}	
	}
	return (res);
}
